/*******************************************************************************
 * Copyright (c) 2012, 2017 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - (#7066) introducing logger & refactoring
 *******************************************************************************/
package org.simantics.document.ui.function;

import java.util.Collection;
import java.util.Collections;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.browser.LocationEvent;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FontDialog;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.request.PossibleResource;
import org.simantics.db.layer0.request.PossibleVariableIndexRoot;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.document.DocumentDialect;
import org.simantics.document.DocumentResource;
import org.simantics.document.DocumentUtils;
import org.simantics.document.ui.Activator;
import org.simantics.document.ui.CSSEditor;
import org.simantics.layer0.Layer0;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.loader.ScenegraphLoaderProcess;
import org.simantics.scenegraph.loader.ScenegraphLoaderUtils;
import org.simantics.scenegraph.loader.ScenegraphLoaderUtils.ScenegraphPropertyReference;
import org.simantics.scl.reflection.annotations.SCLValue;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.scl.runtime.function.FunctionImpl1;
import org.simantics.scl.runtime.function.FunctionImpl3;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.ResourceEditorInput2;
import org.simantics.ui.workbench.action.DefaultActions;
import org.simantics.utils.threads.SWTThread;
import org.simantics.utils.ui.workbench.WorkbenchUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class All {

	private static final Logger LOGGER = LoggerFactory.getLogger(All.class);

	private static Resource getDocumentedResource(ReadGraph graph, Variable context) throws DatabaseException {
		Variable selection = ScenegraphLoaderUtils.getPossibleVariableSelection(graph, context);
		return selection != null ? selection.getPossibleRepresents(graph) : null;
	}

	private static NavigableMap<Double, Resource> getDocumentTypeBindings(ReadGraph graph, Resource forResource, Resource indexRoot) throws DatabaseException {
		DocumentResource DOC = DocumentResource.getInstance(graph);

		Instances query = graph.adapt(DOC.DocumentTypeBinding, Instances.class);
		Collection<Resource> foundBindings = query.find(graph, indexRoot);
		if (foundBindings.isEmpty())
			return Collections.emptyNavigableMap();

		Set<Resource> forResourceTypes = graph.getTypes(forResource);
		TreeMap<Double, Resource> bindings = new TreeMap<>();
		for (Resource binding : foundBindings) {
			double priority = graph.getRelatedValue2(binding, DOC.DocumentTypeBinding_priority, Bindings.DOUBLE);
			Resource type = graph.getSingleObject(binding, DOC.DocumentTypeBinding_HasType);
			if (forResourceTypes.contains(type)) {
				bindings.put(priority, binding);
			}
		}

		return bindings;
	}

	private static boolean createDocument(WriteGraph graph, Resource resource, Resource indexRoot) throws DatabaseException {

		Layer0 L0 = Layer0.getInstance(graph);
		DocumentResource DOC = DocumentResource.getInstance(graph);

		if(graph.hasStatement(resource, DOC.HasDocumentation)) return true;

		NavigableMap<Double, Resource> bindings = getDocumentTypeBindings(graph, resource, indexRoot);
		if(bindings.isEmpty()) return false;

		Resource binding = bindings.lastEntry().getValue();

		Resource documentType = graph.getSingleObject(binding, DOC.DocumentTypeBinding_HasDocumentType);
		Resource document = graph.newResource();
		graph.claim(document, L0.InstanceOf, null, DOC.ScenegraphDocument);
		graph.claimLiteral(document, L0.HasName, "Documentation"); //$NON-NLS-1$
		graph.claim(resource, DOC.HasDocumentation, document);
		graph.claim(document, L0.PartOf, resource);

		Resource scenegraph = graph.newResource();
		graph.claim(scenegraph, L0.InstanceOf, null, documentType);
		graph.claimLiteral(scenegraph, L0.HasName, "Scenegraph"); //$NON-NLS-1$
		graph.claim(scenegraph, L0.PartOf, document);
		graph.claim(document, DOC.ScenegraphDocument_scenegraph, scenegraph);

		return true;

	}

	@SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
	public static Object onCreateDocumentButton(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

		return new FunctionImpl1<Object, Object>() {

			@Override
			public Object apply(Object _o) {

				Simantics.async(new ReadRequest() {

					@Override
					public void run(ReadGraph graph) throws DatabaseException {

						Variable selection = ScenegraphLoaderUtils.getPossibleVariableSelection(graph, context);
						if (selection == null)
							return;

						Resource input = selection.getPossibleRepresents(graph);
						if (input == null)
							return;

						graph.asyncRequest((WriteGraph wg) -> {
							Resource indexRoot = wg.sync(new PossibleVariableIndexRoot(selection));
							if (indexRoot != null)
								createDocument(wg, input, indexRoot);
						});

					}

				});

				return true;

			}

		};

	}

	@SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
	public static Object onDeleteDocumentButton(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

		return new FunctionImpl1<Object, Object>() {

			@Override
			public Object apply(Object _o) {

				Simantics.async(new ReadRequest() {

					@Override
					public void run(ReadGraph graph) throws DatabaseException {

						Resource input = getDocumentedResource(graph, context);
						if (input == null)
							return;

						DocumentResource DOC = DocumentResource.getInstance(graph);
						if (!graph.hasStatement(input, DOC.HasDocumentation))
							return;

						graph.asyncRequest((WriteGraph wg) -> {
							Resource document = wg.getPossibleObject(input, DOC.HasDocumentation);
							if (document != null) {
								RemoverUtil.remove(wg, document);
								wg.markUndoPoint();
							}
						});

					}

				});

				return true;

			}

		};

	}

	@SCLValue(type = "ReadGraph -> Resource -> Variable -> a")
	public static Object editorLoaded(ReadGraph graph, final Resource resource, final Variable context) throws DatabaseException {

		return new FunctionImpl3<WriteGraph, Variable, Variable, Boolean>() {

			@Override
			public Boolean apply(WriteGraph graph, Variable editor, Variable input) {

//				try {
//					return createDocument(graph, input);
//				} catch (DatabaseException e) {
//					Logger.defaultLogError(e);
//				}

				return true;

			}

		};

	}

	@SCLValue(type = "ReadGraph -> Resource -> Variable -> Boolean")
	public static Boolean hasDocument(ReadGraph graph, final Resource resource, final Variable context) throws DatabaseException {

		Resource input = getDocumentedResource(graph, context);
		if (input == null)
			return false;

		DocumentResource DOC = DocumentResource.getInstance(graph);
		return graph.hasStatement(input, DOC.HasDocumentation);

	}

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> a")
    public static Object viewInputChanged(ReadGraph graph, final Resource resource, final Variable context) throws DatabaseException {
    	
    	return new FunctionImpl1<Variable, Boolean>() {

			@Override
			public Boolean apply(final Variable viewVariable) {
				return true;
			}
    		
    	};
    	
    }
    
	
    @SCLValue(type = "ReadGraph -> Variable -> Boolean")
    public static Boolean isWikitext(ReadGraph graph, Variable context) throws DatabaseException {
    	Layer0 L0 = Layer0.getInstance(graph);
    	DocumentResource DOC = DocumentResource.getInstance(graph);
    	Resource p = context.getPossiblePredicateResource(graph);
    	if (p == null)
    	    return Boolean.FALSE;
    	Resource range = graph.getPossibleObject(p, L0.HasRange);
    	return DOC.WikiDocument_WikiText.equals(range);
    }
    
    private static Variable getDefaultSelection(ReadGraph graph, Variable context) throws DatabaseException {
    	
		DocumentResource DOC = DocumentResource.getInstance(graph);
    	Variable sel = ScenegraphLoaderUtils.getVariableSelection(graph, context);
    	
    	Resource represents = sel.getRepresents(graph);
    	Resource doc = graph.getSingleObject(represents, DOC.HasDocument);
    	Resource scenegraph = graph.getSingleObject(doc, DOC.ScenegraphDocument_scenegraph);
    	
    	Variable runtime = ScenegraphLoaderUtils.getRuntimeVariable(graph, context);
		INode root = runtime.adapt(graph, INode.class);

    	Variable result = ScenegraphLoaderProcess.getVariable(graph, null, scenegraph, ScenegraphLoaderUtils.getRuntime(graph, context), root);
    	if(result != null) {
    		Variable userDoc = result.getPossibleProperty(graph, "UserDocumentation"); //$NON-NLS-1$
    		if(userDoc != null) return userDoc;
    	}
    	
    	return null;
    	
    }
    
    private static Variable resolveEditSelection(ReadGraph graph, Variable context, String path) throws DatabaseException {
    	
		final ScenegraphPropertyReference<Variable> selectionReference = 
				ScenegraphLoaderUtils.getRelativePropertyReference(SWTThread.getThreadAccess(), graph, context, path);
		
		Variable editSelection = selectionReference.getExternalValue(graph);
		if(editSelection == null) editSelection = getDefaultSelection(graph, context);
		
		return editSelection;
    	
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> a")
    public static Object wikitextModifier(ReadGraph graph, final Resource resource, final Variable context) throws DatabaseException {
    	
		final ScenegraphPropertyReference<String> textReference = ScenegraphLoaderUtils.getRelativePropertyReference(SWTThread.getThreadAccess(), graph, context, ".../TextContainer/Text#text"); //$NON-NLS-1$
    	
    	return new FunctionImpl1<Object, Object>() {

			@Override
			public Object apply(final Object event) {
    	    
				final String value = textReference.getValue();

				try {
					Simantics.getSession().sync(new WriteRequest() {

						@Override
						public void perform(WriteGraph graph) throws DatabaseException {
							Variable selection = resolveEditSelection(graph, context, "..../Scroll/Browser#edited"); //$NON-NLS-1$
							if (selection != null) {
							    selection.setValue(graph, (String)value, Bindings.STRING);
								graph.markUndoPoint();
							} else {
								LOGGER.error("No selection for resource : " + resource + ", Variable context : " + context + ", value : " + value); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
							}
						}
						
					});
				} catch (DatabaseException e) {
					LOGGER.error("Failed to write wiki text with resource {}, Variable context {}, value {}", resource, context, value, e);
				}
				return null;
			}
    		
    	};
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> String")
    public static String wikitext(ReadGraph graph, Resource resource, Variable context) throws DatabaseException {
    	Variable selection = ScenegraphLoaderUtils.getVariableSelection(graph, context);
    	return selection.getValue(graph);
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> String")
    public static String selectedDocumentPart(ReadGraph graph, Resource resource, Variable context) throws DatabaseException {

		Variable selection = resolveEditSelection(graph, context, ".../Scroll/Browser#edited"); //$NON-NLS-1$
		if(selection == null) return null;

    	return selection.getValue(graph);
    	
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object documentStructureSelected(ReadGraph graph, Resource resource, Variable context) throws DatabaseException {
    	
    	return new FunctionImpl1<Object, Boolean>() {
    		
    	    @Override
    	    public Boolean apply(Object _event) {
    	    	
    	    	Event event = (Event)_event;
    	    	
    	    	final TreeItem item = (TreeItem)event.item;
    	    	NodeContext context = (NodeContext)item.getData();
    	    	final Variable entry = (Variable)context.getConstant(BuiltinKeys.INPUT);
    	    	
    	        ISelectionProvider provider = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActivePart().getSite().getSelectionProvider();
    	        provider.setSelection(new StructuredSelection(entry));
    	        
    	        return null;
    	        
    	    }
    		
    	};
        
    }

    abstract static class WikiButtonModifier extends FunctionImpl1<Object, Object> {

		final ScenegraphPropertyReference<String> textReference;
		final ScenegraphPropertyReference<Point> selectionReference;

    	public WikiButtonModifier(ReadGraph graph, Variable context) throws DatabaseException {
    		textReference = ScenegraphLoaderUtils.getRelativePropertyReference(SWTThread.getThreadAccess(), graph, context, ".../TextContainer/Text#text"); //$NON-NLS-1$
    		selectionReference = ScenegraphLoaderUtils.getRelativePropertyReference(SWTThread.getThreadAccess(), graph, context, ".../TextContainer/Text#selection"); //$NON-NLS-1$
    	}
    	
    	abstract void perform(String before, String selected, String after, Point selection);
    	
	    @Override
	    public Object apply(Object _event) {
	    	String text = textReference.getValue();
	    	Point selection = selectionReference.getValue(); 
	    	String before = text.substring(0, selection.x);
	    	String selected = text.substring(selection.x, selection.y);
	    	String after = text.substring(selection.y);
	    	perform(before, selected, after, selection);
	    	return null;
	    }

    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object boldModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				if(selected.isEmpty()) {
					textReference.setValue(before + "'''bold text'''" + after); //$NON-NLS-1$
				} else {
					textReference.setValue(before + "'''" + selected + "'''" + after);	 //$NON-NLS-1$ //$NON-NLS-2$
				}
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object italicModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				if(selected.isEmpty()) {
					textReference.setValue(before + "''italic text''" + after); //$NON-NLS-1$
				} else {
					textReference.setValue(before + "''" + selected + "''" + after);	 //$NON-NLS-1$ //$NON-NLS-2$
				}
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object strikethroughModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				if(selected.isEmpty()) {
					textReference.setValue(before + "<span style=\"text-decoration:line-through;\">strikethrough text</span>" + after); //$NON-NLS-1$
				} else {
					textReference.setValue(before + "<span style=\"text-decoration:line-through;\">" + selected + "</span>" + after);	 //$NON-NLS-1$ //$NON-NLS-2$
				}
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object underlineModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				if(selected.isEmpty()) {
					textReference.setValue(before + "<span style=\"text-decoration:underline;\">strikethrough text</span>" + after); //$NON-NLS-1$
				} else {
					textReference.setValue(before + "<span style=\"text-decoration:underline;\">" + selected + "</span>" + after);	 //$NON-NLS-1$ //$NON-NLS-2$
				}
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object horizontalRulerModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n<hr/>\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object indentModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + ":" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object fontModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
    		private String hex2(int value) {
    			String result = Integer.toHexString(value);
    			if(result.length() == 1) result = "0" + result; //$NON-NLS-1$
    			return result;
    		}
    		
			@Override
			void perform(String before, String selected, String after, Point selection) {

				FontDialog dialog = new FontDialog(Display.getCurrent().getActiveShell()); 
				FontData data = dialog.open();
				if(data == null) return;

				String family = data.getName();
				int size = data.getHeight();
				
				RGB rgb = dialog.getRGB();
				String hex = hex2(rgb.red) + hex2(rgb.green) + hex2(rgb.blue);

				if(selected.isEmpty()) {
					textReference.setValue(before + "<font style=\"font-size:" + size + ";color: #" + hex + ";font-family:" + family + ";\" >formatted text</font>" + selected + after); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				} else {
					textReference.setValue(before + "<font style=\"font-size:" + size + ";color: #" + hex + ";font-family:" + family + ";\" >" + selected + "</font>" + after); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
				}
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object imageModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
    		
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "[[Image:root://Library/image.png|100px]]" + "\r\n" + selected + after); //$NON-NLS-1$ //$NON-NLS-2$
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object header1Modifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n= Header 1 =\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object header2Modifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n== Header 2 ==\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object header3Modifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n=== Header 3 ===\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object header4Modifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n==== Header 4 ====\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object numberedListModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n" + //$NON-NLS-1$
						"# Item1\r\n" +  //$NON-NLS-1$
						"# Item2\r\n" +  //$NON-NLS-1$
						"## Item2.1\r\n" +  //$NON-NLS-1$
						"# Item3\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object bulletListModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n" + //$NON-NLS-1$
						"* Item1\r\n" +  //$NON-NLS-1$
						"* Item2\r\n" +  //$NON-NLS-1$
						"** Item2.1\r\n" +  //$NON-NLS-1$
						"* Item3\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object tableModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before + "\r\n" + //$NON-NLS-1$
						"{| border=\"1\"\r\n" +  //$NON-NLS-1$
						"! header\r\n" + //$NON-NLS-1$
						"! header2 \r\n" + //$NON-NLS-1$
						"|-\r\n" + //$NON-NLS-1$
						"| cell || cell2\r\n" +  //$NON-NLS-1$
						"|-\r\n" + //$NON-NLS-1$
						"| cell3\r\n" +  //$NON-NLS-1$
						"| cell4\r\n" +  //$NON-NLS-1$
						"|}\r\n"  + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object internalLinkModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {
        return new WikiButtonModifier(graph, context) {
            @Override
            void perform(String before, String selected, String after, Point selection) {
                textReference.setValue(before +
                        "[[Media:root://Documents/Document.pdf|Link to a file within the model]]\r\n" + selected + after); //$NON-NLS-1$
                
            }
        };
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object linkModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				textReference.setValue(before +
						"[http://www.simantics.org External Website Link]\r\n" + selected + after); //$NON-NLS-1$
				
			}
			
		};
        
    }
    
    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Object styleModifier(ReadGraph graph, Resource resource, final Variable context) throws DatabaseException {

    	return new WikiButtonModifier(graph, context) {
			
			@Override
			void perform(String before, String selected, String after, Point selection) {

				Simantics.getSession().asyncRequest(new ReadRequest() {

					@Override
					public void run(ReadGraph graph) throws DatabaseException {

				    	Variable sel = ScenegraphLoaderUtils.getVariableSelection(graph, context);
				    	Resource root = sel.getIndexRoot(graph);
				        String editorId = CSSEditor.EDITOR_ID;
		                RVI rvi = null;

		                PlatformUI.getWorkbench().getDisplay().asyncExec(() -> {
		                    try {
		                        WorkbenchUtils.openEditor(editorId, new ResourceEditorInput2(editorId, root, root, rvi));
		                    } catch (PartInitException e) {
		                        LOGGER.error("Failed to open CSS editor for root " + root, e); //$NON-NLS-1$
		                    }
		                });

					}
					
				});
				
			}
			
		};
        
    }

	@SCLValue(type = "ReadGraph -> Resource -> Variable -> String")
	public static String noDocumentText(ReadGraph graph, final Resource resource, final Variable context) throws DatabaseException {

		Variable selection = ScenegraphLoaderUtils.getPossibleVariableSelection(graph, context);
		if(selection == null) return "<no input>"; //$NON-NLS-1$

		Resource input = selection.getPossibleRepresents(graph);
		if(input == null) return "<no input>"; //$NON-NLS-1$

		String path = DocumentUtils.indexRootPath(graph, selection);
		if(!path.isEmpty()) {
			return "for " + path + selection.getName(graph); //$NON-NLS-1$
		}
		return "for " + NameUtils.getSafeLabel(graph, input); //$NON-NLS-1$

	}

	@SCLValue(type = "ReadGraph -> Resource -> Variable -> Boolean")
	public static Boolean canCreateDocument(ReadGraph graph, final Resource resource, final Variable context) throws DatabaseException {

		Variable selection = ScenegraphLoaderUtils.getPossibleVariableSelection(graph, context);
		if (selection == null)
			return false;

		Resource input = selection.getPossibleRepresents(graph);
		if (input == null)
			return false;

		Resource indexRoot = graph.sync(new PossibleVariableIndexRoot(selection));
		if (indexRoot == null)
			return false;

		NavigableMap<Double, Resource> bindings = getDocumentTypeBindings(graph, input, indexRoot);
		if (bindings.isEmpty())
			return false;

		return true;

	}

    private static class ResolveURI extends UnaryRead<String, Object> {
        public ResolveURI(String uri) {
            super(uri);
        }
        @Override
        public Object perform(ReadGraph graph) throws DatabaseException {
            Object result = graph.syncRequest(new PossibleResource(parameter));
            if (result == null)
                result = Variables.getPossibleVariable(graph, parameter);
            return result;
        }
    }

    private static final Function1<Object, Boolean> PERFORM_DEFAULT_ACTION_FOR_URI_RESOURCE = new Function1<Object, Boolean>() {
        @Override
        public Boolean apply(Object p0) {
            LocationEvent le = (LocationEvent) p0;
            if (le.location.startsWith(DocumentDialect.SIMANTICS_INTERNAL_URI_PREFIX)) {
                // This is not a valid URL anyway so deny relocation.
                le.doit = false;

                // Try to execute default action for the resource or variable
                // that the URI represents.
                String uri = le.location.substring(DocumentDialect.SIMANTICS_INTERNAL_URI_PREFIX.length());
                try {
                    Session s = Simantics.getSession();
                    Object input = RequestUtil.trySyncRequest(s,
                            SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                            SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                            new ResolveURI(uri));
                    if (input != null) {
                        DefaultActions.asyncPerformDefaultAction(s, input, false, false, false);
                    }
                } catch (DatabaseException | InterruptedException e) {
                    Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Failed to resolve URI to a database resource or variable: " + uri, e)); //$NON-NLS-1$
                }
            }
            return true;
        }
    };

    @SCLValue(type = "ReadGraph -> Resource -> Variable -> b")
    public static Function1<Object, Boolean> locationChanging(ReadGraph graph, Resource variable, Variable context) throws DatabaseException {
        return PERFORM_DEFAULT_ACTION_FOR_URI_RESOURCE;
    }

}
