/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.document;

import java.io.File;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;

/**
 * Action for importing files as documents.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ImportDocument extends AddDocumentAction {
	

	public ImportDocument(ReadGraph graph, String relationUri) throws DatabaseException {
		super(graph, relationUri);
	}

	@Override
	public Runnable create(Object target) {
		if(!(target instanceof Resource))
			return null;
		final Resource resource = (Resource)target;
		return new Runnable() {
			
			@Override
			public void run() {
				FileDialog dialog = new FileDialog(Display.getCurrent().getActiveShell(),SWT.OPEN | SWT.MULTI);
				// TODO : is there any way to read file/executable bindings from OS?
				//        if is, use those extensions to filter this list.
				//        note: in windows using "reg query ..." to read bindings form registry would work.
				// Note : If the above mentioned filtering is implemented it should be made optional / configurable.
				dialog.setFilterExtensions(new String[]{"*.*"});
				if (dialog.open() == null) return;

				String filterPath = dialog.getFilterPath();
				String[] filenames = dialog.getFileNames();
				
				ImportJob job = new ImportJob(filenames.length > 1 ? "Import files" : "Import file", resource, filterPath, filenames);
				job.setUser(true);
				job.schedule();
			}
		};
	}
	
	private class ImportJob extends DatabaseJob {

		public ImportJob(String name, Resource resource, String path, String[] filenames) {
			super(name);
			this.resource = resource;
			this.path = path;
			this.filenames = filenames;
		}

		Resource resource;
		String path;
		String[] filenames;

		@Override
		protected IStatus run(final IProgressMonitor monitor) {
			monitor.beginTask("Importing...", filenames.length);
			try {
				Simantics.getSession().syncRequest(new WriteRequest() {
					@Override
					public void perform(WriteGraph graph) throws DatabaseException {
						try {
							graph.markUndoPoint();
						    for (String filename : filenames) {
						    	File f = new File(path, filename);
						    	Resource newDoc = FileDocumentUtil.importFileWithName(graph, f.getAbsolutePath());
						    	linkDocument(graph, resource, newDoc);
							    monitor.worked(1);
						    }
						} catch (Exception e) {
							throw new DatabaseException(e);
						}
					}
				});
				return new Status(IStatus.OK, Activator.PLUGIN_ID, "Import succesful.");
			} catch (DatabaseException e) {
				return new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Import failed.", e);
			} finally {
				monitor.done();
			}
		}
	}
	
}
