/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.document;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleVariableIndexRoot;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.document.function.WikiDocumentNodeImpl;
import org.simantics.document.node.Composite;
import org.simantics.layer0.Layer0;
import org.simantics.scenegraph.loader.ScenegraphLoaderProcess;
import org.simantics.scenegraph.loader.ScenegraphLoaderUtils;
import org.simantics.scenegraph.loader.ScenegraphVariable;
import org.simantics.scenegraph.ontology.ScenegraphResources;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.wiki.ui.SimanticsDialect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.pdf.PdfWriter;

public class DocumentUtils {

	private static final Logger LOGGER = LoggerFactory.getLogger(DocumentUtils.class);

	static class HasDoc extends ResourceRead<Boolean> {
		protected HasDoc(Resource resource) {
			super(resource);
		}

		@Override
		public Boolean perform(ReadGraph graph) throws DatabaseException {
			var DOC = DocumentResource.getInstance(graph);
			return graph.getObjects(resource, DOC.HasDocument).size() > 0;
		}
	}

	static class DeepDocs extends ResourceRead<List<Resource>> {

		protected DeepDocs(Resource resource) {
			super(resource);
		}

		@Override
		public List<Resource> perform(ReadGraph graph) throws DatabaseException {

			DocumentResource DOC = DocumentResource.getInstance(graph);
			Resource doc = graph.getPossibleObject(resource, DOC.HasDocument);
			//LOGGER.debug("DeepDocs({}): doc = {}", NameUtils.getURIOrSafeNameInternal(graph, resource), NameUtils.getURIOrSafeNameInternal(graph, doc));

			Layer0 L0 = Layer0.getInstance(graph);
			Collection<Resource> children = graph.getObjects(resource, L0.ConsistsOf);
			if(children.isEmpty()) {
				if(doc != null) return Collections.singletonList(resource);
				else return Collections.emptyList();
			} else {
				ArrayList<Resource> result = new ArrayList<Resource>();
				if(doc != null) result.add(resource);
				for(Resource child : children)
					result.addAll(graph.syncRequest(new DeepDocs(child)));
				return result;
			}
			
		}

	}

	public DocumentSettings getDocumentSettings(RequestProcessor processor, final Resource resource) throws DatabaseException {

		return processor.syncRequest(new ResourceRead<DocumentSettings>(resource) {

			@Override
			public DocumentSettings perform(ReadGraph graph) throws DatabaseException {

				Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(resource));
				if(indexRoot == null) return DocumentSettings.DEFAULT;
				
				DocumentResource DOC = DocumentResource.getInstance(graph);
				DocumentSettings result = graph.getPossibleRelatedValue(indexRoot, DOC.documentSettings, DocumentSettings.BINDING);
				if(result == null) return DocumentSettings.DEFAULT;
				return result;
				
			}
			
		});

	}

	public void getDocumentCSSText(RequestProcessor processor, final Resource resource, final StringBuilder css) throws DatabaseException {

		css.append(processor.sync(new ResourceRead<String>(resource) {

			@Override
			public String perform(ReadGraph graph) throws DatabaseException {

				Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(resource));
				if(indexRoot == null) return "";
				
				DocumentResource DOC = DocumentResource.getInstance(graph);
				String css = graph.getPossibleRelatedValue(indexRoot, DOC.cssDocument, Bindings.STRING);
				if(css == null) return "";
				
				return css;
				
			}
			
		}));

	}

	public void getDocumentWikiTextRecursive(Session session, final Resource resource, final StringBuilder wiki, final StringBuilder css) throws DatabaseException {
		getDocumentWikiText(session, resource, wiki, css, true);
	}

	public void getDocumentWikiText(Session session, final Resource resource, final StringBuilder wiki, final StringBuilder css, boolean recursive) throws DatabaseException {

		List<Resource> rs = recursive
				? session.sync(new DeepDocs(resource))
				: (session.sync(new HasDoc(resource)) ? Collections.singletonList(resource) : Collections.emptyList());
		for(Resource r : rs) {
			getDocumentWikiText(session, r, wiki);
		}
		
		getDocumentCSSText(session, resource, css);

	}

	public void getDocumentWikiText(Session session, final Resource resource, final StringBuilder b) {

		ScenegraphLoaderProcess loader = new ScenegraphLoaderProcess(new Composite(), "CreatePDFAction");

		try {

			final Variable context = session.sync(new WriteResultRequest<Variable>(session.getService(VirtualGraph.class)) {

				@Override
				public Variable perform(WriteGraph graph) throws DatabaseException {

					DocumentResource DOC = DocumentResource.getInstance(graph);
					Resource doc = graph.getPossibleObject(resource, DOC.HasDocument);
					if(doc == null) return null;

					Layer0 L0 = Layer0.getInstance(graph);
					ScenegraphResources SG = ScenegraphResources.getInstance(graph);
					Resource runtime = graph.newResource();
					graph.claim(runtime, L0.InstanceOf, null, SG.Runtime);
					Variable base = Variables.getVariable(graph, resource);

					String uri = base.getURI(graph);
					graph.claimLiteral(runtime, SG.Runtime_HasVariable, uri, Bindings.STRING);

					return new ScenegraphVariable(base, doc, runtime, loader.getRoot());

				}

			});

			if(context == null) return;

			String wiki = session.sync(new UniqueRead<String>() {

				@Override
				public String perform(ReadGraph graph) throws DatabaseException {

					DocumentResource DOC = DocumentResource.getInstance(graph);
					Resource doc = graph.getSingleObject(resource, DOC.HasDocumentation);
					Resource scenegraph = graph.getSingleObject(doc, DOC.ScenegraphDocument_scenegraph);

					WikiDocumentNodeImpl node = loader.load(graph, scenegraph, ScenegraphLoaderUtils.getRuntime(graph, context));

					StringBuilder bb = new StringBuilder();
					node.create(bb, true);
					String text = bb.toString();

					return SimanticsDialect.INSTANCE.apply(graph, Variables.getVariable(graph, resource), text);

				}

			});

			b.append(wiki);

		} catch (DatabaseException e) {
			LOGGER.error("getDocumentWikiText failed unexpectedly", e);
		} finally {
			loader.dispose();
		}
	}

	public int print(RequestProcessor processor, Resource res, String wiki, String css, DocumentSettings settings, final PdfWriter writer, final Document document)
			throws DocumentException, DatabaseException {
		Exportable exp = processor.syncRequest(new UniqueRead<Exportable>() {
			@Override
			public Exportable perform(ReadGraph graph) throws DatabaseException {
				return new Exportable(graph, res, wiki, css, settings, true);
			}
		});
		return exp.export(document, writer);
	}

	public static String indexRootPath(ReadGraph graph, Variable selection) throws DatabaseException {

		Variable possibleConfiguration = Variables.getPossibleConfigurationVariable(graph, selection);
		if(possibleConfiguration != null) selection = possibleConfiguration;

		Resource indexRoot = graph.syncRequest(new PossibleVariableIndexRoot(selection));
		if(indexRoot == null) return "";

		Variable selectionParent = selection.getParent(graph);
		if (selectionParent == null) return "";

		String selectionURI = selectionParent.getURI(graph);
		String selectionRoleIdentifier = selection.getRole(graph).getIdentifier();
		String suffix = " " + selectionRoleIdentifier + " ";

		SimulationResource SIMU = SimulationResource.getInstance(graph);
		Resource configuration = graph.getPossibleObject(indexRoot, SIMU.HasConfiguration);
		if(configuration != null) {
			String configurationURI = graph.getURI(configuration);
			if(selectionURI.startsWith(configurationURI)) {
				if(selectionURI.equals(configurationURI)) return "Configuration / ";
				return URIStringUtils.unescape(selectionURI.substring(configurationURI.length() + 1).replace("/", " / ")) + suffix;
			}
		}

		String rootURI = graph.getURI(indexRoot);
		String result = selectionURI.replace(rootURI, "");
		result += selectionRoleIdentifier;
		return URIStringUtils.unescape(result);

	}

}
