/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.connection.tests;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.HeadlessException;
import java.awt.RenderingHints;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;

import javax.swing.JApplet;

import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.connection.RouteLine;
import org.simantics.diagram.connection.RouteLink;
import org.simantics.diagram.connection.RouteTerminal;
import org.simantics.diagram.connection.actions.IAction;
import org.simantics.diagram.connection.actions.IReconnectAction;
import org.simantics.diagram.connection.actions.MoveAction;
import org.simantics.diagram.connection.actions.ReconnectLineAction;
import org.simantics.diagram.connection.rendering.ExampleConnectionStyle;
import org.simantics.diagram.connection.rendering.IRouteGraphRenderer;
import org.simantics.diagram.connection.rendering.StyledRouteGraphRenderer;
import org.simantics.diagram.connection.rendering.arrows.ArrowExampleLineEndStyle;
import org.simantics.diagram.connection.rendering.arrows.ILineEndStyle;
import org.simantics.diagram.connection.rendering.arrows.PlainExampleLineEndStyle;
import org.simantics.diagram.connection.tests.actions.HighlightReconnectPointsAction;
import org.simantics.diagram.connection.tests.actions.NewTerminalAction;

public class TestApplet extends JApplet {

	private static final long serialVersionUID = 6135978753249808640L;
	
	public static final double TOLERANCE = IAction.TOLERANCE;
    
    RouteGraph rg;
    IRouteGraphRenderer renderer = 
            new StyledRouteGraphRenderer(new ExampleConnectionStyle());
    
    double posX = 0.0;
    double posY = 0.0;
    
    double mouseX;
    double mouseY;
    
    IAction currentAction;
    
    public TestApplet() {
        this(new RouteGraph());
        
        RouteLine v = rg.addLine(false, 0);
                
        rg.link(addTerminal(rg, -100, -100, 1, PlainExampleLineEndStyle.INSTANCE), v);
        rg.link(addTerminal(rg, -100, 100, 1, PlainExampleLineEndStyle.INSTANCE), v);
        rg.link(addTerminal(rg, 100, -80, 1, ArrowExampleLineEndStyle.INSTANCE), v);
        rg.link(addTerminal(rg, 100, 100, 4, ArrowExampleLineEndStyle.INSTANCE), v);
    }
    
    public TestApplet(RouteGraph rg_) throws HeadlessException {
        this.rg = rg_;        
        
        addMouseListener(new MouseAdapter() {
            
            @Override
            public void mousePressed(MouseEvent e) {
                if(e.getButton() == MouseEvent.BUTTON1) {
                    mouseX = e.getX() - posX;
                    mouseY = e.getY() - posY;
                    
                    if(e.isAltDown()) {
                        currentAction = new NewTerminalAction(rg, mouseX, mouseY);
                        repaint();
                    }
                    else if(currentAction instanceof NewTerminalAction) {
                        ((NewTerminalAction)currentAction).finish(mouseX, mouseY);
                        currentAction = null;
                        repaint();
                    }
                    else if(currentAction instanceof HighlightReconnectPointsAction) {
                        currentAction = ReconnectLineAction.create(rg, mouseX, mouseY);
                        repaint();
                    }
                    else if(currentAction instanceof IReconnectAction) {
                        ((IReconnectAction)currentAction).finish(mouseX, mouseY);
                        currentAction = null;
                        repaint();
                    }
                    else {
                        currentAction = MoveAction.create(rg, mouseX, mouseY);
                    }
                }
            }
            
            @Override
            public void mouseReleased(MouseEvent e) {
                if(currentAction instanceof MoveAction) {
                    ((MoveAction)currentAction).finish(mouseX, mouseY);
                    currentAction = null;
                    repaint();
                }
            }
        });
        
        addMouseMotionListener(new MouseMotionListener() {          
            @Override
            public void mouseMoved(MouseEvent e) {
                mouseX = e.getX() - posX;
                mouseY = e.getY() - posY;
                
                if(currentAction != null)
                    repaint();
            }
            
            @Override
            public void mouseDragged(MouseEvent e) {
                mouseX = e.getX() - posX;
                mouseY = e.getY() - posY;
                
                if(currentAction != null)
                    repaint();
            }
        });
        
        addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                Object target = rg.pick(mouseX, mouseY, TOLERANCE);
                
                if(e.getKeyCode() == KeyEvent.VK_COMMA) {
                    if(target instanceof RouteTerminal) {
                        RouteTerminal terminal = (RouteTerminal)target;
                        rg.rotate(terminal, -1);
                        repaint();
                    }                   
                }
                else if(e.getKeyCode() == KeyEvent.VK_PERIOD) {
                    if(target instanceof RouteTerminal) {
                        RouteTerminal terminal = (RouteTerminal)target;
                        rg.rotate(terminal, 1);
                        repaint();
                    }                   
                }   
                else if(e.getKeyCode() == KeyEvent.VK_D) {
                    if(target instanceof RouteTerminal) {
                        RouteTerminal terminal = (RouteTerminal)target;
                        rg.toggleDirectLines(terminal);
                        repaint();
                    }                   
                }  
                else if(e.getKeyCode() == KeyEvent.VK_S) {
                    if(target instanceof RouteLine) {
                        RouteLine rLine = (RouteLine)target;
                        rg.split(rLine, rLine.isHorizontal() ? mouseX : mouseY);
                        repaint();
                    }
                }
                else if(e.getKeyCode() == KeyEvent.VK_DELETE) {
                    if(target instanceof RouteLine) {
                        RouteLine line = (RouteLine)target;
                        rg.merge(line);                     
                    }
                    else if(target instanceof RouteLink) {
                        rg.deleteCorner((RouteLink)target);
                    }
                    else if(target instanceof RouteTerminal) {
                        rg.remove((RouteTerminal)target);
                    }
                    repaint();
                }
                else if(e.getKeyCode() == KeyEvent.VK_ESCAPE) {
                    if(currentAction != null)
                        currentAction = null;
                    repaint();
                }
                else if(e.getKeyCode() == KeyEvent.VK_P) {
                    rg.print();
                }
                else if(e.getKeyCode() == KeyEvent.VK_CONTROL) {
                    if(currentAction == null)
                        currentAction = new HighlightReconnectPointsAction(rg);
                    repaint();
                }
            }            
            
            @Override
            public void keyReleased(KeyEvent e) {
                if(e.getKeyCode() == KeyEvent.VK_CONTROL) {
                    if(currentAction instanceof HighlightReconnectPointsAction)
                        currentAction = null;
                    repaint();
                }
            }
            
        });
        
    }
    
    private static RouteTerminal addTerminal(RouteGraph rg, double x, double y, int allowedDirections, ILineEndStyle style) {
        return rg.addTerminal(x, y, x-20, y-20, x+20, y+20, allowedDirections, style);
    }
    
    @Override
    public void paint(Graphics g_) {
        Graphics2D g = (Graphics2D)g_ ;
        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g.setColor(Color.BLACK);
        g.setBackground(Color.WHITE);
        g.clearRect(0, 0, getWidth(), getHeight());
        posX = getWidth()*0.5;
        posY = getHeight()*0.5;
        g.translate(posX, posY);
        
        if(currentAction != null)
            currentAction.render(g, renderer, mouseX, mouseY);
        else
            renderer.render(g, rg);
    }
}
