/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.service;

import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.request.WriteTraits;
import org.simantics.utils.datastructures.Pair;

/**
 * @author TUOKSK
 *
 */
public interface XSupport {
    /**
     * Writes illegal data into the current cluster cluster stream.
     *
     * @param r is resource used to identify cluster. If null then all clusters
     *        streams are corrupted.
     */
    void corruptCluster(Resource r)
    throws DatabaseException;
    /**
     * Creates new cluster proxy for given clusterId.
     * Depending on the cluster id this proxy is valid or invalid.
     *
     * @param clusterId is used to identify cluster.
     * @return index of cluster proxy in cluster table.
     * @throws DatabaseException
     */
    int corruptClusterTable(long clusterId)
    throws DatabaseException;
    /**
     * Flush all modified data in the given cluster stream to server.
     *
     * @param r is resource used to identify cluster. If null then all clusters
     *        streams are flushed.
     */
    void flushCluster(Resource r)
    throws ServiceException;

    /**
     * Forcibly breaks underlying connection.
     */
    void breakConnection()
    throws DatabaseException;
    /**
     * Set cluster stream on or off.
     *
     * @param setOff true if cluster stream is turned off
     * @throws DatabaseException
     */
    void setClusterStreamOff(boolean setOff)
    throws DatabaseException;

    /**
     * Clears the meta data cache.
     *
     * @return count of meta data entries in cache before clearing
     * @throws DatabaseException
     */
    int clearMetadataCache()
    throws DatabaseException;
    /**
     * Sends commit message to server but does not end transaction.
     * @param wgraph Write graph to ensure that this is called inside transaction.
     *
     */
    <T> void commitAndContinue(WriteOnlyGraph wograph, WriteTraits wtraits);

    /**
     * Get cluster's immutable property.
     *
     * @param resource used to get cluster.
     * @return true if cluster is immutable.
     * @throws DatabaseException
     */
    boolean getImmutable(Resource resource)
    throws DatabaseException;
    /**
     * Set cluster's immutable property.
     *
     * @param resource used to get cluster.
     * @param immutable value to set.
     * @throws DatabaseException
     */
    void setImmutable(Resource resource, boolean immutable)
    throws DatabaseException;

    /**
     * @param allowImmutableWrites <code>true</code> to allow writing to immutable cluster 
     * @param createAsImmutable <code>true</code> to make all created clusters immutable
     */
    void setServiceMode(boolean allowImmutableWrites, boolean createAsImmutable);
    Pair<Boolean,Boolean> getServiceMode();

    /**
     * If resource is acquired from DelayedWriteGraph during delayed write request then
     * this method can be used to convert it to normal resource which can be use in
     * write and write only requests.
     *
     * @param resource to convert.
     * @return converted resource if it was acquired from delayed write request.
     * Otherwise returns input resource.
     */
    Resource convertDelayedResourceToResource(Resource resource);

    /**
     * Execute given command on server.
     * @param command
     * @return result of the command
     * @throws DatabaseException
     */
    String execute(String command)
    throws DatabaseException;
    /**
     * Tests cluster functionality.
     * Database session is needed but no layer0.
     * Enables us to test cluster implementation without layer0.
     *
     * @param session
     */
    void testCluster(Session session)
    throws DatabaseException;

    /**
     * Writes garbage to server page table and exits server with -1 exit code.
     *
     * @throws DatabaseException
     */
    public void corruptPageTableAndDie()
    throws DatabaseException;

    /**
     * Returns clusters known to server.
     * @throws DatabaseException
     */
    public ClusterUID[] listClusters()
    throws DatabaseException;

    /**
     * Deletes given cluster from server.
     * @param clusterUID
     * @throws DatabaseException
     */
    public void deleteCluster(ClusterUID clusterUID)
    throws DatabaseException;
    public void purge() throws DatabaseException;

    public boolean rolledback();
    
    public boolean isClusterLoaded(ClusterUID clusterUID) throws DatabaseException;

    public void saveQueries() throws DatabaseException;

    public WriteTraits getWriteRequest(ReadGraph graph) throws DatabaseException;

    public void snapshot() throws DatabaseException;

    /*
     * Invalidates all build-in queries for the given resources.
     * Required for working with immutable updates. 
     * @since 1.58.0
     */
    public void invalidateRequests(WriteOnlyGraph graph, Collection<Resource> resource);

}
