/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import java.util.Collection;
import java.util.Properties;
import java.util.function.Consumer;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.ClusterUID;

/**
 * The interface that every driver class must implement.
 *
 * <P>
 * Each driver should supply a class that implements the Driver interface.
 *
 * <P>
 * The {@link Manager} will try to load as many drivers as it can find and then
 * for any given connection request, it will ask each driver in turn to try to
 * connect to the target URL.
 *
 * <P>
 * It is strongly recommended that each Driver class should be small and
 * standalone so that the Driver class can be loaded and queried without
 * bringing in vast quantities of supporting code.
 *
 * <P>
 * When a Driver class is loaded, it should create an instance of itself and
 * register it with the DriverManager. This means that a user can load and
 * register a driver by calling
 *
 * <pre>
 *   &lt;code&gt;Class.forName(&quot;foo.bah.Driver&quot;)&lt;/code&gt;
 * </pre>
 *
 * @see Manager
 */
public interface Driver {
    /**
     * Driver name identifies the used database driver.
     * At any time there can be only one implementation registered for each name.
     *
     * @return name of the driver instance.
     */
    public String getName();
    /**
     * @return current database user agent. Can be null.
     */
    public DatabaseUserAgent getDatabaseUserAgent(String address) throws DatabaseException;
    /**
     * Set user agent to be used by other methods. Can be null.
     *
     * @param dbUserAgent defines error recover policy. Database specific.
     */
    public void setDatabaseUserAgent(String address, DatabaseUserAgent dbUserAgent) throws DatabaseException;
    /**
     * Attempts to make a database connection to the given address.
     *
     * <P>
     * The driver should throw a {@link DatabaseException} if it has trouble
     * connecting to the database.
     *
     * <P>
     * The <code>java.util.Properties</code> argument can be used to pass
     * arbitrary string tag/value pairs as connection arguments. Normally at
     * least "user" and "password" properties should be included in the
     * <code>Properties</code> object.
     *
     * Properties:<ol>
     *   <li>user</li>
     *   <li>password</li>
     * </ol>
     *
     * @param address of the database session. Driver implementation specifies the address format.
     * @param properties a list of arbitrary string tag/value pairs as connection
     *        arguments. Normally at least a "user" and "password" property
     *        should be included.
     * @return a <code>Session</code> object that represents a connection to the
     *         database.
     * @throws DatabaseException TODO: separate possible errors (connection,
     *         authentication, initialization)
     */
    Session getSession(String address, Properties properties) throws DatabaseException;

    /**
     * Return database server that can be used to control a local database process.
     *
     * @param address of the database server. Driver implementation specifies the address format.
     * @param properties a list of arbitrary string tag/value pairs as connection
     *        arguments. Normally at least a "user" and "password" property
     *        should be included.
     * @return ServerI
     * @throws DatabaseException
     */
    ServerI getServer(String address, Properties properties) throws DatabaseException;

    /**
     * Return database management object that can be used to control a local database.
     *
     * @param address of the database. Driver implementation specifies the address format.
     * @param properties a list of arbitrary string tag/value pairs as connection
     *        arguments. Normally at least a "user" and "password" property
     *        should be included.
     * @return Management object for controlling database.
     * @throws DatabaseException
     */
    Management getManagement(String address, Properties properties) throws DatabaseException;

    interface Management {
        /**
         * Existence of database means that the database has been created.
         *
         * @return true if database exist.
         *
         * @throws DatabaseException
         */
        public boolean exist() throws DatabaseException;
        /**
         * Delete's database. After this {@link #exist()} must return false.
         *
         * @throws DatabaseException if database deletion failed.
         * This may leave the database in an inconsistent state.
         * Recovery from this is implementation specific.
         */
        public void delete() throws DatabaseException;
        /**
         * Creates a new database. After this {@link #exist()} must return false.
         *
         * @throws DatabaseException if database creation failed.
         * This may leave the database in an inconsistent state.
         * Recovery from this is implementation specific.
         */
        public void create() throws DatabaseException;
        /**
         * Removes caching data leaving only persistent data.
         *
         * @throws DatabaseException
         */
        public void purge(Consumer<Collection<ClusterUID>> callback) throws DatabaseException;
        public void shutdown() throws DatabaseException;
    }
}
