/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.server.protocol;

import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;

public class DataBuffer {
    private static final Charset UTF8 = Charset.forName("UTF-8");
    private boolean DEBUG = false;
    private ByteBuffer buffer = null;
    enum Allocation { JavaAllocation, DirectAllocation }
    DataBuffer(Allocation a) {
        switch (a) {
        case JavaAllocation:
             buffer = ByteBuffer.allocate(20);
            break;
        case DirectAllocation:
            buffer = ByteBuffer.allocate(20);
            break;
        }
    }
    DataBuffer(byte[] bytes, int size) {
        buffer = ByteBuffer.allocate(size);
        buffer.put(bytes, 0, size);
        buffer.rewind();
    }
    DataBuffer(byte[] bytes) {
        buffer = ByteBuffer.wrap(bytes);
        buffer.rewind();
    }
    public DataBuffer(ByteBuffer byteBuffer) {
        buffer = byteBuffer.duplicate();
        buffer.order(byteBuffer.order());
    }
    public DataBuffer(ByteBuffer byteBuffer, int dummy) {
        buffer = byteBuffer;
    }
    public ByteBuffer getByteBuffer() {
        return buffer;
    }
    private void checkCapacity(int a) {
        if (buffer.capacity() - buffer.position() >= a)
            return;
        ByteBuffer t = buffer;
        int capacity = t.capacity();
        int position = t.position();
        int newCapacity = capacity + Math.max(capacity/2, a);
        buffer = ByteBuffer.allocate(newCapacity);
        t.clear();
        buffer.put(t);
        buffer.position(position);
        buffer.order(t.order());
    }
    public void clear() {
        buffer.clear();
    }
    public void order(ByteOrder byteOrder) {
        buffer.order(byteOrder);
    }
    public void mark() {
        buffer.mark();
    }
    public void position(int position) {
        buffer.position(position);
    }
    public byte[] getBytes() {
        byte[] t = new byte[buffer.position()];
        buffer.clear();
        buffer.get(t);
        return t;
    }
    public short get(short a) {
        return buffer.getShort();
    }
    public int get(int a) {
        return buffer.getInt();
    }
    public void put(int a) {
        checkCapacity(4);
        buffer.putInt(a);
    }
    public int[] get(int[] a) {
        int size = buffer.getInt();
        int[] t = new int[size];
        buffer.asIntBuffer().get(t);
        buffer.position(buffer.position() + size * 4);
        return t;
    }
    public void put(int[] a) {
        if (null == a)
            a = new int[0];
        checkCapacity(4 + 4*a.length);
        this.put(a.length);
        for (int i=0; i<a.length; ++i)
            buffer.putInt(a[i]);
    }
    public long[] get(long[] a) {
        int size = buffer.getInt();
        long[] t = new long[size];
        buffer.asLongBuffer().get(t, 0, size);
        buffer.position(buffer.position() + 8 * size);
        return t;
    }
    public void put(long[] a) {
        checkCapacity(4 + 8*a.length);
        this.put(a.length);
        for (int i=0; i<a.length; ++i)
            buffer.putLong(a[i]);
    }
    public boolean get(boolean a) {
        byte b = buffer.get();
        return !(0 == b);
    }
    public void put(boolean a) {
        checkCapacity(1);
        byte b = a ? (byte)0xff : (byte)0;
        buffer.put(b);
    }
    public byte get(byte a) {
        return buffer.get();
    }
    public void put(byte a) {
        checkCapacity(1);
        buffer.put(a);
    }
    public byte[] get(byte[] a) {
        int size = buffer.getInt();
        byte[] t = new byte[size];
        buffer.get(t, 0, size);
        return t;
    }
    public void put(byte[] a) {
        checkCapacity(4 + a.length);
        this.put(a.length);
        buffer.put(a);
    }
    public ByteBuffer get(ByteBuffer a) {
        int size = buffer.getInt();
        byte[] t = new byte[size];
        buffer.get(t, 0, size);
        a.put(t);
        return a;
    }
    public void put(ByteBuffer a) {
        byte[] t = a.array();
        checkCapacity(4 + t.length);
        this.put(t.length);
        buffer.put(t);
    }
    public static void printChars(PrintStream out, ByteBuffer buf, int pos) {
        out.print("[" + buf.limit() + "]");
        for(int i=pos;i<buf.limit();++i) {
            int val = (int)buf.get(i);
            if(val < 0)
                val += 256;
            char c = (char)val;
            if(c >= 32 && c < 128)
                out.print(c);
            else if(c==0)
                out.print('\u00A4');
            else
                out.print("(" + val + ")");
        }
        out.println();
    }
    public String get(String a) {
        byte[] t = null;
        t = this.get(t);
        CharsetDecoder decoder = UTF8.newDecoder();
        ByteBuffer bbuf = ByteBuffer.wrap(t);
        CharBuffer cbuf;
        String s = null;
        try {
            cbuf = decoder.decode(bbuf);
            s = cbuf.toString();
        } catch (CharacterCodingException e) {
            bbuf.rewind();
            if (DEBUG)
                printChars(System.err, bbuf, 0);
            try {
                cbuf = UTF8
                .newDecoder()
                .onMalformedInput(CodingErrorAction.REPLACE)
                .onUnmappableCharacter(CodingErrorAction.REPLACE)
                .decode(bbuf);
                s = cbuf.toString();
            } catch (CharacterCodingException e1) {
                throw new Error("not possible", e1);
            }
        }
        return s;
    }
    public void put(String a) {
        try {
            put(a.getBytes(UTF8.name()));
        } catch (UnsupportedEncodingException e) {
            throw new Error("UnsupportedEncoding: " + UTF8.name());
        }
    }
    public long get(long a) {
        return buffer.getLong();
    }
    public void put(long a) {
        checkCapacity(8);
        buffer.putLong(a);
    }
    public float get(float f) {
        return buffer.getFloat();
    }
    public void put(float a) {
        checkCapacity(4);
        buffer.putFloat(a);
    }
    public double get(double f) {
        return buffer.getDouble();
    }
    public void put(double a) {
        checkCapacity(8);
        buffer.putDouble(a);
    }
    public String[] get(String[] a) {
        int size = buffer.getInt();
        String[] t = new String[size];
        for (int i=0; i<size; ++i)
            t[i] = this.get(t[i]);
        return t;
    }
    public void put(String[] a) {
        checkCapacity(4);
        this.put(a.length);
        for (int i=0; i<a.length; ++i)
            this.put(a[i]);
    }
}
