package org.simantics.db.layer0.variable;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.function.All;

public class ConstantPropertyVariable extends AbstractPropertyVariable {

    final protected Variable parent;
    final String name;
    final Object value;
    final Binding binding;
    final Map<String, Variable> properties;
    final private Resource predicate;

    public ConstantPropertyVariable(Variable parent, String name, Object value, Binding binding, Resource predicate, Collection<ConstantPropertyVariableBuilder> propertyBuilders, Set<String> classifications) {
        assert parent != null;
        assert name != null;
        this.parent = parent;
        this.name = name;
        this.value = value;
        this.binding = binding;
        this.predicate = predicate;
        this.properties = new HashMap<String, Variable>(propertyBuilders.size());
        for(ConstantPropertyVariableBuilder builder : propertyBuilders) {
            properties.put(builder.getName(), builder.build(this));
        }
        if(!classifications.isEmpty())
            properties.put(Variables.CLASSIFICATIONS, new ConstantPropertyVariable(this, Variables.CLASSIFICATIONS, classifications, null));
    }

    public ConstantPropertyVariable(Variable parent, String name, Object value, Binding binding, Collection<ConstantPropertyVariableBuilder> propertyBuilders, Set<String> classifications) {
        this(parent, name, value, binding, null, Collections.<ConstantPropertyVariableBuilder>emptyList(), Collections.<String>emptySet());
    }

    public ConstantPropertyVariable(Variable parent, String name, Object value, Binding binding) {
        this(parent, name, value, binding, Collections.<ConstantPropertyVariableBuilder>emptyList(), Collections.<String>emptySet());
    }
  
    @Override
    public <T> T getValue(ReadGraph graph) throws DatabaseException {
        return (T)value;
    }

    @Override
    public <T> T getValue(ReadGraph graph, Binding binding)
            throws DatabaseException {        
        return (T)value;
    }

    @Override
    public void setValue(WriteGraph graph, Object value, Binding binding)
            throws DatabaseException {
        throw new DatabaseException("Value is constant.");
    }

    @Override
    public String getName(ReadGraph graph) throws DatabaseException {
        return name;
    }

    @Override
    public Resource getType(ReadGraph graph) throws DatabaseException {
        return null;
    }

    @Override
    public Resource getPossibleType(ReadGraph graph) throws DatabaseException {
        return null;
    }

    @Override
    public Resource getType(ReadGraph graph, Resource baseType) throws DatabaseException {
        return null;
    }

    @Override
    public Resource getPossibleType(ReadGraph graph, Resource baseType) throws DatabaseException {
        return null;
    }

//    @Override
//    public Object getSerialized(ReadGraph graph) throws DatabaseException {
//        return name;
//    }

    @Override
    public Variable getParent(ReadGraph graph) throws DatabaseException {
        return parent;
    }

    @Override
    public Resource getRepresents(ReadGraph graph) throws DatabaseException {
        return null;
    }
    
    @Override
    public Datatype getDatatype(ReadGraph graph) throws DatabaseException {
        return binding != null ? binding.type() : null;
    }
    
    @Override
    public Resource getPropertyResource(ReadGraph graph) throws DatabaseException {
    	return predicate;
    }

    @Override
    public Resource getContainerResource(ReadGraph graph) throws DatabaseException {
    	return null;
    }
    
    @Override
    public Variable getPredicate(ReadGraph graph) throws DatabaseException {
    	return null;
    }
    
    @Override
    protected Variable getPossibleDomainProperty(ReadGraph graph, String name)
    		throws DatabaseException {
    	Variable result = properties.get(name);
    	if(result != null) return result;
    	if(predicate != null) {
    		HashMap<String,Variable> ps = new HashMap<String,Variable>();
    		All.collectPropertiesFromContext(graph, this, predicate, ps);
    		return ps.get(name);
    	}
    	return null; 
    }

    @Override
    public Map<String, Variable> collectDomainProperties(ReadGraph graph,
    		Map<String, Variable> properties) throws DatabaseException {
    	if(!this.properties.isEmpty()) {
    		if(properties == null) properties = new HashMap<String,Variable>(this.properties.size());
        	properties.putAll(this.properties);
    	}
    	if(predicate != null) All.collectPropertiesFromContext(graph, this, predicate, properties);
    	return properties;
    }
    
    @Override
    public Set<String> getClassifications(ReadGraph graph) throws DatabaseException {
    	Variable property = getPossibleDomainProperty(graph, Variables.CLASSIFICATIONS);
    	if(property != null) return property.getValue(graph);
    	else return Collections.emptySet();
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + name.hashCode();
        result = prime * result + parent.hashCode();
        result = prime * result + ((value == null) ? 0 : value.hashCode());
        result = prime * result + ((binding == null) ? 0 : binding.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ConstantPropertyVariable other = (ConstantPropertyVariable) obj;
        if (!name.equals(other.name))
            return false;
        if (!parent.equals(other.parent))
            return false;
        if (!Objects.equals(value, other.value))
            return false;
        return Objects.equals(binding, other.binding);
    }

}
