package org.simantics.db.layer0.scl;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.scl.CompileValueRequest.CompilationContext;
import org.simantics.db.layer0.util.RuntimeEnvironmentRequest;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.environment.Environments;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.top.SCLExpressionCompilationException;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function1;

/**
 * Compiles an SCL expression that is attached to a literal
 * whose parent is a component that is a part of a component type.
 * 
 * @author Tuukka Lehtonen
 */
public class CompileValueRequest extends AbstractExpressionCompilationRequest<CompilationContext, Object> {

    public static class CompilationContext extends AbstractExpressionCompilationContext {
        public CompilationContext(RuntimeEnvironment runtimeEnvironment) {
            super(runtimeEnvironment);
        }
    }

    protected final Resource relation;
    protected final Resource literal;

    public CompileValueRequest(Resource literal, Resource relation) {
        this.relation = relation;
        this.literal = literal;
    }

    public CompileValueRequest(ReadGraph graph, Variable context) throws DatabaseException {
        this(context.getRepresents(graph), context.getPredicateResource(graph));
    }

    public static Object compileAndEvaluate(ReadGraph graph, Variable context) throws DatabaseException {
        SCLContext sclContext = SCLContext.getCurrent();
        Object oldGraph = sclContext.get("graph");
        try {
            Function1<Object,Object> exp = graph.syncRequest(new CompileValueRequest(graph, context),
                    TransientCacheListener.instance());
            sclContext.put("graph", graph);
            return exp.apply(context);
        } catch (DatabaseException e) {
            throw (DatabaseException)e;
        } catch (Throwable t) {
            throw new DatabaseException(t);
        } finally {
            sclContext.put("graph", oldGraph);
        }
    }

    public static Function1<Object,Object> compile(ReadGraph graph, Resource literal, Resource predicate) throws DatabaseException {
        SCLContext sclContext = SCLContext.getCurrent();
        Object oldGraph = sclContext.get("graph");
        try {
            Function1<Object,Object> exp = graph.syncRequest(new CompileValueRequest(literal, predicate), TransientCacheListener.instance());
            sclContext.put("graph", graph);
            return exp;
        } catch (DatabaseException e) {
            throw (DatabaseException)e;
        } catch (Throwable t) {
            throw new DatabaseException(t);
        } finally {
            sclContext.put("graph", oldGraph);
        }
    }

    @Override
    protected String getExpressionText(ReadGraph graph)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        return graph.getRelatedValue(literal, L0.SCLValue_expression, Bindings.STRING);
    }

    protected Resource getIndexRoot(ReadGraph graph) throws DatabaseException {
        return graph.syncRequest(new IndexRoot(literal));
    }

    @Override
    protected Type getExpectedType(ReadGraph graph, CompilationContext context)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        String valueType = graph.getPossibleRelatedValue(relation, L0.RequiresValueType, Bindings.STRING);
        if(valueType != null) {
            try {
                return Environments.getType(context.runtimeEnvironment.getEnvironment(), valueType);
            } catch (SCLExpressionCompilationException e) {
                e.printStackTrace();
            }
        }
        return super.getExpectedType(graph, context);
    }

    @Override
    protected CompilationContext getCompilationContext(ReadGraph graph)
            throws DatabaseException {
        Resource indexRoot = getIndexRoot(graph);
        RuntimeEnvironment runtimeEnvironment = graph.syncRequest(new RuntimeEnvironmentRequest(indexRoot));
        return new CompilationContext(runtimeEnvironment);
    }

    @Override
    protected Type getContextVariableType() {
        return VARIABLE; 
    }

    @Override
    protected Expression getVariableAccessExpression(
            ReadGraph graph,
            CompilationContext context,
            org.simantics.scl.compiler.elaboration.expressions.Variable contextVariable,
            String name) throws DatabaseException {
        if(name.equals("self"))
            return new EVariable(contextVariable);
        else
            return null;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((literal == null) ? 0 : literal.hashCode());
        result = prime * result + ((relation == null) ? 0 : relation.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        CompileValueRequest other = (CompileValueRequest) obj;
        if (literal == null) {
            if (other.literal != null)
                return false;
        } else if (!literal.equals(other.literal))
            return false;
        if (relation == null) {
            if (other.relation != null)
                return false;
        } else if (!relation.equals(other.relation))
            return false;
        return true;
    }


}
