/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.indexing.internal;

import java.util.concurrent.Semaphore;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.jobs.Job;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.function.DbConsumer;
import org.simantics.db.indexing.exception.IndexingException;

/**
 * @author Tuukka Lehtonen
 * @since 1.22.2, 1.25.0
 */
public abstract class IndexingJob extends Job {

    /**
     * NOTE: this is intentionally the same as
     * <code>org.simantics.DatabaseJob.DATABASE_JOB_FAMILY</code> in order for
     * this job to cause the same UI-behavior as DatabaseJob.
     */
    private static final String DATABASE_JOB_FAMILY = "org.simantics.db.inDatabaseJob";

    public IndexingJob(String name) {
        super(name);
    }

    @Override
    public boolean belongsTo(Object family) {
        return DATABASE_JOB_FAMILY.equals(family);
    }

    /**
     * @param monitor
     * @param jobName
     * @param consumer
     * @throws DatabaseException
     */
    public static void jobifyIfPossible(
            IProgressMonitor monitor,
            String jobName,
            DbConsumer<IProgressMonitor> consumer)
                    throws DatabaseException
    {
        // Prevent deadlocks by checking preconditions for using a job.
        // JobManager is suspended e.g. during workbench startup.
        if (Job.getJobManager().isSuspended() || Job.getJobManager().currentJob() != null) {
            consumer.accept(monitor);
            return;
        }

        Semaphore barrier = new Semaphore(0);
        Throwable[] err = { null };
        IndexingJob job = new IndexingJob(jobName) {
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                try {
                    consumer.accept(monitor);
                } catch (Throwable t) {
                    err[0] = t;
                } finally {
                    monitor.done();
                    barrier.release();
                }
                return org.eclipse.core.runtime.Status.OK_STATUS;
            }
        };
        job.schedule();
        try {
            barrier.acquire();
            if (err[0] != null) {
                if (err[0] instanceof IndexingException)
                    throw (DatabaseException) err[0];
                throw new IndexingException(err[0]);
            }
        } catch (InterruptedException e) {
            throw new IndexingException(e);
        }
    }

}
