/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;

import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

public final class SuperRelations extends UnaryQueryPIntSet {

    public SuperRelations(final int resource) {
        super(resource);
    }

    static final SuperRelations entry(final QueryProcessor provider, final int r) {
        return (SuperRelations)provider.cache.superRelationsMap.get(r);
    }

 	@Override
 	public final void removeEntry(QueryProcessor provider) {
 		provider.cache.remove(this);
 	}
 	
    static class Ints {
        
        private TIntHashSet set = null;
        public int single = 0;
        
        public boolean add(int val) {
            if(single == val) return false;
            if(single == 0) {
                single = val;
                return true;
            }
            if(set == null) set = new TIntHashSet(4);
            return set.add(val);
        }
        
        public int size() {
            
            if(single == 0) return 0;
            if(set == null) return 1;
            return set.size() + 1;
            
        }

        public void forEach(TIntProcedure proc) {
            if(single > 0) proc.execute(single);
            if(set != null) set.forEach(proc);
        }
        
    }

 	public void compute(final ReadGraphImpl graph, final InternalProcedure<IntSet> procedure) throws DatabaseException {
 	    computeForEach(graph, id, this, procedure);
 	}

    static class DirectProcedure extends Ints implements IntProcedure, TIntProcedure, InternalProcedure<IntSet> {
        
        IntSet result;
        InternalProcedure<IntSet> proc;
        
        public DirectProcedure(IntSet result, InternalProcedure<IntSet> proc) {
            this.result = result;
            this.proc = proc;
        }
        
        @Override
        final public boolean execute(int r) {
            result.add(r);
            return true;
        }
        @Override
        final public void execute(ReadGraphImpl graph, int r) {
            if(single == 0) {
                single = r;
                return;
            }
            add(r);
        }
        @Override
        public final void execute(ReadGraphImpl graph, IntSet set) throws DatabaseException {
            set.forEach(this);
            proc.execute(graph, result);
        }
        @Override
        public void finished(ReadGraphImpl graph) {
        }
        @Override
        public void exception(ReadGraphImpl graph, Throwable t) {
            throw new Error("Errors are not supported.", t);
        }

    }

 	public static Object computeForEach(final ReadGraphImpl graph, int id, SuperRelations entry, final InternalProcedure<IntSet> procedure_) throws DatabaseException {
 	    
 	    InternalProcedure<IntSet> procedure = entry != null ? entry : procedure_;

 	    QueryProcessor processor = graph.processor;
 		
 		processor.querySupport.ensureLoaded(graph, id);
 		
 	    final InternalProcedure<IntSet> proc = (InternalProcedure<IntSet>)procedure;

 	    final int subrelationOf = processor.getSubrelationOf();

 	    final IntSet result = new IntSet(processor.querySupport);

 	    final DirectProcedure directProc = new DirectProcedure(result, proc);

 	   processor.querySupport.getObjects(graph, id, subrelationOf, directProc);

 	    int size = directProc.size();

 	    if(size == 0) {
 	    	proc.execute(graph, IntSet.EMPTY);
 	    } else if (size == 1) {
 	    	result.add(directProc.single);
 	    	QueryCache.runnerSuperRelations(graph, directProc.single, entry, null, directProc);
 	    } else {

 	    	final TIntProcedure addToResult = new TIntProcedure() {
 	    		@Override
 	    		public boolean execute(int r) {
 	    			synchronized(result) {
 	    				result.add(r);
 	    			}
 	    			return true;
 	    		}
 	    	};

 	    	final AtomicInteger finishes = new AtomicInteger(0);

 	    	directProc.forEach(new TIntProcedure() {

 	    		@Override
 	    		public boolean execute(int arg0) {
 	    			try {
						return execute0(arg0);
					} catch (DatabaseException e) {
						Logger.defaultLogError(e);
					}
 	    			return false;
 	    		}

 	    		public boolean execute0(int arg0) throws DatabaseException {

 	    			synchronized(result) {
 	    				result.add(arg0);
 	    			}

 	    			QueryCache.runnerSuperRelations(graph, arg0, entry, null, new InternalProcedure<IntSet>() {

 	    				@Override
 	    				public void execute(ReadGraphImpl graph, IntSet set) throws DatabaseException {
 	    					set.forEach(addToResult);
 	    					int current = finishes.addAndGet(1);
 	    					if(current == directProc.size()) {
 	    						proc.execute(graph, result);
 	    						return; 
 	    					}
 	    				}

 	    				@Override
 	    				public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
 	    					proc.exception(graph, t);
 	    				}

 	    			});

 	    			return true;

 	    		}

 	    	});

 	    }
 	    
 	    if(entry != null) entry.performFromCache(graph, procedure_);
 	     
 	    return result;
        
    }
    
    @Override
    public String toString() {
    	return "SuperRelations[" + id + "]";
    }
    
}
