/*******************************************************************************
 * Copyright (c) 2018, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import org.simantics.databoard.Bindings;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.debug.ListenerReport;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.utils.Development;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.collections.CollectionUtils;
import org.slf4j.LoggerFactory;

import gnu.trove.ext.IdentityHashSet;
import gnu.trove.map.hash.THashMap;
import gnu.trove.set.hash.THashSet;

public class QueryListening {

    static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(QueryListening.class);

    private final QueryProcessor                                    processor;
    private final Scheduler                                         scheduler;
    private final Map<ListenerBase,ListenerEntry>                   addedEntries = new HashMap<>();

    private THashSet<ListenerEntry>                                 scheduledListeners    = new THashSet<ListenerEntry>();
    private boolean                                                 firingListeners       = false;

    final Map<CacheEntry, Object>                                   parents = new IdentityHashMap<>();

    final THashMap<CacheEntry, ArrayList<ListenerEntry>>            listeners = new THashMap<>(10, 0.75f);

    QueryListening(QueryProcessor processor) {
        
        this.processor = processor;
        scheduler = new Scheduler(processor);
        scheduler.start();
        
    }

    public boolean hasScheduledUpdates() {
        return !scheduledListeners.isEmpty();
    }
    
    void stopThreading() {
        try {
            scheduler.sync();
            int check = scheduler.singleThreadRequested.getAndIncrement();
            if(check > 0)
                LOGGER.error("Problem in query listening bookkeeping", new Exception());
        } catch (Throwable t) {
            LOGGER.error("Error while waiting for query dependency management", t);
        }
    }

    void startThreading() {
        try {
            int check = scheduler.singleThreadRequested.decrementAndGet();
            if(check > 0)
                LOGGER.error("Problem in query listening bookkeeping", new Exception());
        } catch (Throwable t) {
            LOGGER.error("Error while waiting for query dependency management", t);
        }
    }

    void registerDependencies(ReadGraphImpl graph, CacheEntry child, CacheEntry parent, ListenerBase listener, Object procedure, boolean inferred) {

        if(inferred) {
            assert(listener == null);
            return;
        }

        if(parent != null) {
            try {
                if(!child.isImmutable(graph) && !parent.isImmutable(graph))
                    scheduler.accept(new RegisterParentRunnable(graph.processor.listening, parent, child));
            } catch (DatabaseException e) {
                LOGGER.error("Error while registering query dependencies", e);
            }
        }

        if(listener != null)
            if(!listener.safeIsDisposed())
                scheduler.accept(new RegisterListenerRunnable(this, listener, procedure, parent, child));

    }

    void registerFirstKnown(ListenerBase base, Object result) {
        
        if(base == null) return;

        scheduler.accept(new RegisterFirstKnownRunnable(addedEntries, base, result));

    }

    void scheduleListener(ListenerEntry entry) {
        
        assert (entry != null);
        
        if (Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_LISTENERS, Bindings.BOOLEAN)) {
                System.err.println("Scheduled " + entry.procedure);
            }
        }
        
        scheduledListeners.add(entry);
        
    }

    boolean hasListener(CacheEntry entry) {
        if(listeners.get(entry) != null) return true;
        return false;
    }

    boolean hasListenerAfterDisposing(CacheEntry entry) {
        ArrayList<ListenerEntry> entries = listeners.get(entry); 
        if(entries != null) {
            ArrayList<ListenerEntry> list = null;
            // Avoid allocating iterators
            for (int i=0, n=entries.size(); i < n; i++) {
                ListenerEntry e = entries.get(i);
                if (e.base.safeIsDisposed()) {
                    if(list == null)
                        list = new ArrayList<>();
                    list.add(e);
                }
            }
            if(list != null) {
                // Avoid allocating iterators
                for (int i=0, n=list.size(); i < n; i++) {
                    entries.remove(list.get(i));
                }
            }
            if (entries.isEmpty()) {
                listeners.remove(entry);
                return false;
            }
            return true;
        }
        return false;
    }

    void processListenerReport(CacheEntry<?> entry, Map<CacheEntry, Set<ListenerBase>> workarea) {

        if(!workarea.containsKey(entry)) {

            HashSet<ListenerBase> ls = new HashSet<ListenerBase>();
            for(ListenerEntry e : getListenerEntries(entry))
                ls.add(e.base);

            workarea.put(entry, ls);

            forParents(entry, parent -> {
                processListenerReport(parent, workarea);
                ls.addAll(workarea.get(parent));
            });

        }

    }

    public synchronized ListenerReport getListenerReport() throws IOException {

        class ListenerReportImpl implements ListenerReport {

            Map<CacheEntry, Set<ListenerBase>> workarea = new HashMap<CacheEntry, Set<ListenerBase>>();

            @Override
            public void print(PrintStream b) {
                Map<ListenerBase, Integer> hist = new HashMap<ListenerBase, Integer>();
                for(Map.Entry<CacheEntry, Set<ListenerBase>> e : workarea.entrySet()) {
                    for(ListenerBase l : e.getValue()) {
                        Integer i = hist.get(l);
                        hist.put(l, i != null ? i-1 : -1);
                    }
                }

                for(Pair<ListenerBase, Integer> p : CollectionUtils.valueSortedEntries(hist)) {
                    b.print("" + -p.second + " " + p.first + "\n");
                }

                b.flush();
            }

        }

        ListenerReportImpl result = new ListenerReportImpl();

        Collection<CacheEntryBase> all = processor.allCaches(new CacheCollectionResult()).toCollection();
        for(CacheEntryBase entry : all) {
            hasListenerAfterDisposing(entry);
        }
        for(CacheEntryBase entry : all) {
            processListenerReport(entry, result.workarea);
        }

        return result;

    }

    public synchronized String reportListeners(File file) throws IOException {

        if (!processor.isAlive())
            return "Disposed!";

        PrintStream b = new PrintStream(new BufferedOutputStream(new FileOutputStream(file)));
        ListenerReport report = getListenerReport();
        report.print(b);

        return "Done reporting listeners.";

    }

    public void fireListeners(ReadGraphImpl graph) {
        stopThreading();
        ReadGraphImpl listenerGraph = graph.forSyncExecute();
        fireListeners_(listenerGraph);
        listenerGraph.asyncBarrier.dec();
        listenerGraph.asyncBarrier.waitBarrier(this, listenerGraph);
        startThreading();
    }

    private void fireListeners_(ReadGraphImpl graph) {

        assert (!processor.updating);
        assert (!processor.cache.collecting);
        assert (!firingListeners);

        firingListeners = true;

        try {

            // Performing may cause further events to be scheduled.
            while (!scheduledListeners.isEmpty()) {

                // Clone current events to make new entries possible during
                // firing.
                THashSet<ListenerEntry> entries = scheduledListeners;
                scheduledListeners = new THashSet<ListenerEntry>();

                ArrayList<ListenerEntry> schedule = new ArrayList<ListenerEntry>();

                for (ListenerEntry listenerEntry : entries) {

                    if (pruneListener(listenerEntry)) {
                        if (Development.DEVELOPMENT) {
                            if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_LISTENERS, Bindings.BOOLEAN)) {
                                new Exception().printStackTrace();
                                System.err.println("Pruned " + listenerEntry.procedure);
                            }
                        }
                        continue;
                    }

                    final CacheEntry entry = listenerEntry.entry;
                    assert (entry != null);

                    Object newValue = processor.compareTo(graph, entry, listenerEntry.getLastKnown());

                    if (newValue != ListenerEntry.NOT_CHANGED) {
                        if (Development.DEVELOPMENT) {
                            if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_LISTENERS, Bindings.BOOLEAN)) {
                                new Exception().printStackTrace();
                                System.err.println("Add to schedule " + listenerEntry.procedure + " with " + newValue);
                            }
                        }
                        schedule.add(listenerEntry);
                        listenerEntry.setLastKnown(entry.getResult());
                    }

                }

                for(ListenerEntry listenerEntry : schedule) {
                    final CacheEntry entry = listenerEntry.entry;
                    if (Development.DEVELOPMENT) {
                        if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_LISTENERS, Bindings.BOOLEAN)) {
                            System.err.println("Firing " + listenerEntry.procedure);
                        }
                    }
                    try {
                        if (Development.DEVELOPMENT) {
                            if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_LISTENERS, Bindings.BOOLEAN)) {
                                System.err.println("Firing " + listenerEntry.procedure + " for " + listenerEntry.entry);
                            }
                        }
                        entry.performFromCache(graph, listenerEntry.procedure);
                    } catch (Throwable t) {
                        LOGGER.debug("Unexpected exception ", t);
                    }
                }

            }

            Set<ExternalReadEntry<?>> externalUpdates = processor.updatedPrimitivesInCurrentWrite;
            processor.updatedPrimitivesInCurrentWrite = new HashSet<>();

            for (ExternalReadEntry<?> query : externalUpdates) {
                pruneParentSet(query);
                // Parents for immediate_update queries are only pruned by the query garbage collector.
                // This means no listeners can exist if query.hasParents() is false.
                if (!query.isDiscarded() && !processor.isBound(query)) {
                    processor.cache.externalReadEntryMap.remove(query.id);
                    query.discard();
                }
            }

        } finally {
            firingListeners = false;
        }

    }

    void updateParents(int indent, CacheEntry entry, Deque<UpdateEntry> todo) {
        forParents(entry, parent -> {
            if(!parent.isDiscarded()) {
                todo.push(new UpdateEntry(entry, parent, indent + 2));
            }
        });
    }

    void updateParent(int indent, CacheEntry entry, CacheEntry parent, Deque<UpdateEntry> todo) {
        if(!parent.isDiscarded())
            todo.push(new UpdateEntry(entry, parent, indent + 2));
     }

    private boolean pruneListener(ListenerEntry entry) {
        
        if (entry.base.safeIsDisposed()) {
            
            assert (entry != null);
            ArrayList<ListenerEntry> list = listeners.get(entry.entry);
            if(list != null) {
                boolean success = list.remove(entry);
                assert (success);
                if (list.isEmpty())
                    listeners.remove(entry.entry);
            }

            return true;
            
        } else {
            
            return false;
            
        }
    }

    private List<ListenerEntry> getListenerEntries(CacheEntry entry) {
        hasListenerAfterDisposing(entry);
        if(listeners.get(entry) != null)
            return listeners.get(entry);
        else 
            return Collections.emptyList();
    }

    private static final CacheEntry REMOVED = new CacheEntryBase<Object>() {
        @Override
        int makeHash() {
            return 0;
        }
        @Override
        Query getQuery() {
            return null;
        }
        @Override
        Object performFromCache(ReadGraphImpl graph, Object procedure) throws DatabaseException {
            return null;
        }
    };

    public void addParent(CacheEntry entry, CacheEntry parent) {
        Object result = parents.get(entry);
        if(result == null) {
            parents.put(entry, parent);
            return;
        }
        if(result instanceof CacheEntry) {
            CacheEntry single = (CacheEntry)result;
            if(single.isDiscarded()) {
                parents.put(entry, parent);
            } else {
                IdentityHashSet<CacheEntry> set = new IdentityHashSet<>(3, CacheEntry.class, REMOVED);
                set.add(single);
                set.add(parent);
                parents.put(entry, set);
            }
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result;
            set.add(parent);
        }
    }

    public void discarded(CacheEntry entry) {
        parents.remove(entry);
    }

    private static List<CacheEntry> NO_PARENTS_ARRAY = List.of();
    
    public List<CacheEntry> getParents(CacheEntry entry) {
        Object result = parents.get(entry);
        if(result == null)
            return NO_PARENTS_ARRAY;
        if(result instanceof CacheEntry) {
            return List.of((CacheEntry)result);
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result; 
            ArrayList<CacheEntry> list = new ArrayList<>(set.size());
            set.values(list::add);
            return list;
        }
    }
    
    public boolean hasParents(CacheEntry entry) {
        return parents.containsKey(entry);
    }
    
    public void forParents(CacheEntry entry, Consumer<CacheEntry> consumer) {
        Object result = parents.get(entry);
        if(result == null)
            return;
        if(result instanceof CacheEntry) {
            consumer.accept((CacheEntry)result);
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result; 
            set.values(consumer);
        }
    }
    
    public CacheEntry getFirstParent(CacheEntry entry) {
        Object result = parents.get(entry);
        if(result == null)
            return null;
        if(result instanceof CacheEntry) {
            return (CacheEntry)result; 
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result;
            return set.firstBy(parent -> !parent.isDiscarded());
        }
    }
    
    public CacheEntry firstParentNotDiscarded(CacheEntry entry) {
        Object result = parents.get(entry);
        if(result == null)
            return null;
        if(result instanceof CacheEntry) {
            CacheEntry parent = (CacheEntry)result;
            if(parent.isDiscarded())
                return null;
            return parent;
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result;
            return set.removeOrGetByPredicate(CacheEntry::isDiscarded);
        }
    }

    public void pruneParentSet(CacheEntry entry) {
        Object result = parents.get(entry);
        if(result == null)
            return;
        if(result instanceof CacheEntry) {
            CacheEntry parent = (CacheEntry)result;
            if(parent.isDiscarded())
                parents.remove(entry);
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result;
            int size = set.removeBy(CacheEntry::isDiscarded);
            if(size == 1) {
                parents.put(entry, set.first());
            } else if(size == 0) {
                parents.remove(entry);
            }
        }
    }

    public boolean removeParent(CacheEntry entry, CacheEntry parent) {
        Object result = parents.get(entry);
        if(result == null)
            return false;
        if(result instanceof CacheEntry) {
            if(parent == result) {
                parents.remove(entry);
                return true;
            }
            return false;
        } else {
            IdentityHashSet<CacheEntry> set = (IdentityHashSet<CacheEntry>)result;
            boolean removed = set.remove(parent);
            if(removed) {
                int size = set.size();
                if(size == 1) {
                    parents.put(entry, set.first());
                } else if(size == 0) {
                    parents.remove(entry);
                }
                return true;
            } else {
                return false;
            }
        }
    }

    private static class RegisterParentRunnable implements Runnable {

        private final QueryListening listening;
        private final CacheEntry parent;
        private final CacheEntry child;

        public RegisterParentRunnable(QueryListening listening, CacheEntry parent, CacheEntry child) {
            this.listening = listening;
            this.parent = parent;
            this.child = child;
        }

        @Override
        public void run() {
            listening.addParent(child, parent);
            if (Development.DEVELOPMENT) {
                if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_DEPENDENCIES, Bindings.BOOLEAN)) {
                    System.out.println(child + " -> " + parent);
                }
            }
        }

    }

    private static class RegisterListenerRunnable implements Runnable {

        private final QueryListening queryListening;
        private final ListenerBase base;
        private final Object procedure;
        private final CacheEntry parent;
        private final CacheEntry entry;

        public RegisterListenerRunnable(QueryListening queryListening, ListenerBase base, Object procedure, CacheEntry parent, CacheEntry entry) {
            this.queryListening = queryListening;
            this.base = base;
            this.procedure = procedure;
            this.parent = parent;
            this.entry = entry;
        }

        @Override
        public void run() {

            assert (entry != null);
            assert (procedure != null);

            ArrayList<ListenerEntry> list = queryListening.listeners.get(entry);
            if (list == null) {
                list = new ArrayList<>(1);
                queryListening.listeners.put(entry, list);
            }

            ListenerEntry result = new ListenerEntry(entry, base, procedure);
            // Equals is here based on base
            int currentIndex = list.indexOf(result);
            // There was already a listener
            if(currentIndex > -1) {
                ListenerEntry current = list.get(currentIndex);
                if(!current.base.safeIsDisposed())
                    return;
                list.set(currentIndex, result);
            } else {
                list.add(result);
            }

            if (Development.DEVELOPMENT) {
                if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_LISTENERS, Bindings.BOOLEAN)) {
                    new Exception().printStackTrace();
                    System.err.println("addListener -> " + list.size() + " " + entry + " " + base + " " + procedure);
                }
            }

            queryListening.addedEntries.put(base, result);

        }


    }

    private static class RegisterFirstKnownRunnable implements Runnable {

        private final Map<ListenerBase,ListenerEntry> addedEntries;
        private final ListenerBase base;
        private final Object result;

        public RegisterFirstKnownRunnable(Map<ListenerBase,ListenerEntry> addedEntries, ListenerBase base, Object result) {
            this.addedEntries = addedEntries;
            this.base = base;
            this.result = result;
        }

        @Override
        public void run() {
            ListenerEntry entry = addedEntries.remove(base);
            if(entry != null) entry.setLastKnown(result);
        }

    }

}
