/*******************************************************************************
 * Copyright (c) 2018, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Semaphore;
import java.util.function.Consumer;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.ReadGraph;
import org.simantics.db.RelationInfo;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.impl.query.QueryProcessor.SessionTask;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;
import org.simantics.db.request.ReadExt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.ext.DualIdContainerHashMapWithPrimaryIdLookup;
import gnu.trove.ext.IdContainerHashMap;
import gnu.trove.ext.StableHashMap;
import gnu.trove.map.hash.THashMap;
import gnu.trove.map.hash.TObjectIntHashMap;

public class QueryCacheBase {

	protected static final Logger LOGGER = LoggerFactory.getLogger(QueryCacheBase.class);

	// Statistics
	final int THREADS;
	public final int THREAD_MASK;
	int                                             hits                  = 0;
	int                                             misses                = 0;
	int                                             updates               = 0;
	public int                                              size                  = 0;

	public volatile boolean dirty = false;
	public boolean collecting = false;

	protected static final boolean SINGLE = true;
	
	protected final IdContainerHashMap<DirectPredicates>                       directPredicatesMap;
	protected final IdContainerHashMap<DirectSuperRelations>                   directSuperRelationsMap;
	protected final IdContainerHashMap<ValueQuery>                             valueQueryMap;
	protected final IdContainerHashMap<ChildMap>                               childMapMap;
	protected final IdContainerHashMap<RelationInfoQuery>                      relationInfoQueryMap;
	protected final IdContainerHashMap<SuperTypes>                             superTypesMap;
	protected final IdContainerHashMap<SuperRelations>                         superRelationsMap;
	protected final IdContainerHashMap<Types>                                  typesMap;
	protected final IdContainerHashMap<PrincipalTypes>                         principalTypesMap;
	protected final IdContainerHashMap<TypeHierarchy>                          typeHierarchyMap;
	protected final IdContainerHashMap<OrderedSet>                             orderedSetMap;
	protected final IdContainerHashMap<Predicates>                             predicatesMap;
	protected final IdContainerHashMap<AssertedPredicates>                     assertedPredicatesMap;

	protected final DualIdContainerHashMapWithPrimaryIdLookup<Objects>                            objectsMap;
	protected final DualIdContainerHashMapWithPrimaryIdLookup<DirectObjects>                      directObjectsMap;
	protected final DualIdContainerHashMapWithPrimaryIdLookup<Statements>                         statementsMap;
	protected final DualIdContainerHashMapWithPrimaryIdLookup<AssertedStatements>                 assertedStatementsMap;

	protected final StableHashMap<AsyncRead, AsyncReadEntry>                  asyncReadEntryMap; 
	protected final StableHashMap<Read, ReadEntry>                            readEntryMap;
	protected final StableHashMap<AsyncMultiRead, AsyncMultiReadEntry>        asyncMultiReadEntryMap; 
	protected final StableHashMap<MultiRead, MultiReadEntry>                  multiReadEntryMap; 
	protected final StableHashMap<ExternalRead, ExternalReadEntry>            externalReadEntryMap; 

	protected final THashMap<String, URIToResource>                           uRIToResourceMap;

	public final QuerySupport                                                 querySupport;

	public QueryCacheBase(QuerySupport querySupport, int threads) {

		THREADS = threads;
		THREAD_MASK = threads - 1;

		this.querySupport = querySupport;
		
		try {
			
			directPredicatesMap = new IdContainerHashMap<DirectPredicates>(DirectPredicates.class);
			directSuperRelationsMap = new IdContainerHashMap<DirectSuperRelations>(DirectSuperRelations.class);
			valueQueryMap = new IdContainerHashMap<ValueQuery>(ValueQuery.class);
			childMapMap = new IdContainerHashMap<ChildMap>(ChildMap.class);
			relationInfoQueryMap = new IdContainerHashMap<RelationInfoQuery>(RelationInfoQuery.class);
			superTypesMap = new IdContainerHashMap<SuperTypes>(SuperTypes.class);
			superRelationsMap = new IdContainerHashMap<SuperRelations>(SuperRelations.class);
			typesMap = new IdContainerHashMap<Types>(Types.class);
			principalTypesMap = new IdContainerHashMap<PrincipalTypes>(PrincipalTypes.class);
			typeHierarchyMap = new IdContainerHashMap<TypeHierarchy>(TypeHierarchy.class);
			orderedSetMap = new IdContainerHashMap<OrderedSet>(OrderedSet.class);
			predicatesMap = new IdContainerHashMap<Predicates>(Predicates.class);
			assertedPredicatesMap = new IdContainerHashMap<AssertedPredicates>(AssertedPredicates.class);
			
			objectsMap = new DualIdContainerHashMapWithPrimaryIdLookup<Objects>(Objects.class);
			directObjectsMap = new DualIdContainerHashMapWithPrimaryIdLookup<DirectObjects>(DirectObjects.class);
			statementsMap = new DualIdContainerHashMapWithPrimaryIdLookup<Statements>(Statements.class);
			assertedStatementsMap = new DualIdContainerHashMapWithPrimaryIdLookup<AssertedStatements>(AssertedStatements.class);
			
			asyncReadEntryMap = new StableHashMap<AsyncRead, AsyncReadEntry>(); 
			readEntryMap = new StableHashMap<Read, ReadEntry>();
			asyncMultiReadEntryMap = new StableHashMap<AsyncMultiRead, AsyncMultiReadEntry>(); 
			multiReadEntryMap = new StableHashMap<MultiRead, MultiReadEntry>(); 
			externalReadEntryMap = new StableHashMap<ExternalRead, ExternalReadEntry>();
			
			uRIToResourceMap = new THashMap<String, URIToResource>();
			
		} catch (Exception e) {
			throw new Error("Fatal error in internal query configuration", e);
		}

	}

	public <T> Object performQuery(ReadGraphImpl parentGraph, final AsyncMultiRead<T> query, final CacheEntryBase entry_, Object procedure_) throws DatabaseException {

		ReadGraphImpl queryGraph = parentGraph.withParent(entry_, null, false);

		AsyncMultiReadEntry entry = (AsyncMultiReadEntry)entry_;
		AsyncMultiProcedure<T> procedure = (AsyncMultiProcedure<T>)procedure_;

		try {

			query.perform(queryGraph, new AsyncMultiProcedure<T>() {

				@Override
				public void execute(AsyncReadGraph graph, T result) {
					ReadGraphImpl impl = (ReadGraphImpl)graph;
					entry.addOrSet(result);
					try {
						procedure.execute(parentGraph, result);
					} catch (Throwable t) {
						t.printStackTrace();
					}
				}

				@Override
				public void finished(AsyncReadGraph graph) {
					ReadGraphImpl impl = (ReadGraphImpl)graph;
					entry.finish(parentGraph);
					try {
						procedure.finished(parentGraph);
					} catch (Throwable t) {
						t.printStackTrace();
					}
				}

				@Override
				public void exception(AsyncReadGraph graph, Throwable t) {
					ReadGraphImpl impl = (ReadGraphImpl)graph;
					entry.except(parentGraph, t);
					try {
						procedure.exception(parentGraph, t);
					} catch (Throwable t2) {
						t2.printStackTrace();
					}
				}

			});

			return entry.getResult();

		} catch (Throwable t) {

			entry.except(t);
			try {
				procedure.exception(parentGraph, t);
			} catch (Throwable t2) {
				t2.printStackTrace();
			}

			return entry.getResult();

		} finally {
			
			queryGraph.asyncBarrier.dec();
			
		}

	}

	public <T> Object performQuery(ReadGraphImpl parentGraph, final MultiRead<T> query, final CacheEntryBase entry_, Object procedure_) throws DatabaseException {

		ReadGraphImpl queryGraph = parentGraph.withParent(entry_, null, true);

		MultiReadEntry entry = (MultiReadEntry)entry_;
		SyncMultiProcedure<T> procedure = (SyncMultiProcedure<T>)procedure_;

		try {

			query.perform(queryGraph, new SyncMultiProcedure<T>() {

				@Override
				public void execute(ReadGraph graph, T result) {
					ReadGraphImpl impl = (ReadGraphImpl)graph;
					entry.addOrSet(result);
					try {
						procedure.execute(parentGraph, result);
					} catch (Throwable t) {
						t.printStackTrace();
					}
				}

				@Override
				public void finished(ReadGraph graph) {
					ReadGraphImpl impl = (ReadGraphImpl)graph;
					entry.finish(parentGraph);
					try {
						procedure.finished(parentGraph);
					} catch (Throwable t) {
						t.printStackTrace();
					}
				}

				@Override
				public void exception(ReadGraph graph, Throwable t) {
					ReadGraphImpl impl = (ReadGraphImpl)graph;
					entry.except((DatabaseException)t);
					try {
						procedure.exception(parentGraph, t);
					} catch (Throwable t2) {
						t2.printStackTrace();
					}
				}

			});

			return entry.getResult();

		} catch (Throwable t) {

			entry.except(t);
			try {
				procedure.exception(parentGraph, t);
			} catch (Throwable t2) {
				t2.printStackTrace();
			}

			return entry.getResult();

		}
		
	}

	private void collectEntries(Consumer<CacheEntryBase<?>> consumer) {

		directPredicatesMap.values(consumer);
		directSuperRelationsMap.values(consumer);
		valueQueryMap.values(consumer);
		childMapMap.values(consumer);
		relationInfoQueryMap.values(consumer);
		superTypesMap.values(consumer);
		superRelationsMap.values(consumer);
		typesMap.values(consumer);
		principalTypesMap.values(consumer);
		typeHierarchyMap.values(consumer);
		orderedSetMap.values(consumer);
		predicatesMap.values(consumer);
		assertedPredicatesMap.values(consumer);
		
		objectsMap.values(consumer);
		directObjectsMap.values(consumer);
		statementsMap.values(consumer);
		assertedStatementsMap.values(consumer);
		
		for(AsyncReadEntry<?> entry : asyncReadEntryMap.values())
			consumer.accept(entry);
		for(ReadEntry<?> entry : readEntryMap.values())
			consumer.accept(entry);
		for(AsyncMultiReadEntry<?> entry : asyncMultiReadEntryMap.values())
			consumer.accept(entry);
		for(MultiReadEntry<?> entry : multiReadEntryMap.values())
			consumer.accept(entry);
		for(ExternalReadEntry<?> entry : externalReadEntryMap.values())
			consumer.accept(entry);
		
		for(URIToResource entry : uRIToResourceMap.values())
			consumer.accept(entry);

	}
	
	public Collection<CacheEntry<?>> getRootList() {

		Collection<CacheEntry<?>> result = new ArrayList<>();
		collectEntries(e -> result.add(e));
		return result;

	}

	public int calculateCurrentSize() {

		int realSize = 0;

		realSize += directPredicatesMap.size();
		realSize += directSuperRelationsMap.size();
		realSize += valueQueryMap.size();
		realSize += childMapMap.size();
		realSize += relationInfoQueryMap.size();
		realSize += superTypesMap.size();
		realSize += superRelationsMap.size();
		realSize += typesMap.size();
		realSize += principalTypesMap.size();
		realSize += typeHierarchyMap.size();
		realSize += orderedSetMap.size();
		realSize += predicatesMap.size();
		realSize += assertedPredicatesMap.size();
		
		realSize += objectsMap.size();
		realSize += directObjectsMap.size();
		realSize += statementsMap.size();
		realSize += assertedStatementsMap.size();

		realSize += asyncReadEntryMap.size();
		realSize += readEntryMap.size();
		realSize += asyncMultiReadEntryMap.size();
		realSize += multiReadEntryMap.size();
		realSize += externalReadEntryMap.size();

		realSize += uRIToResourceMap.size();
		
		return realSize;

	}
	
	CacheCollectionResult allCaches(CacheCollectionResult result) {
		collectEntries(result);
		return result;
	}

	public void scanPending() {

		ArrayList<CacheEntry> entries = new ArrayList<>();
		
		collectEntries(entries::add);
		
		for(Object e : entries) {
			if(e instanceof CacheEntry) {
				CacheEntry en = (CacheEntry)e;
				if(en.isPending()) System.out.println("pending " + e);
				if(en.isExcepted()) System.out.println("excepted " + e);
				if(en.isDiscarded()) System.out.println("discarded " + e);
				if(en.isRefuted()) System.out.println("refuted " + e);
				if(en.isFresh()) System.out.println("fresh " + e);
			} else {
				//System.out.println("Unknown object " + e);
			}
		}
	}

	public static void waitPending(ReadGraphImpl graph, CacheEntry entry) throws DatabaseException {

		int counter = 0;
		while(entry.isPending()) {
			try {
			    boolean performed = false;//graph.performPending();
			    if(!performed) {
					Thread.sleep(1);
					counter++;
					if(counter > 30000) {
						CacheEntryBase base = ((CacheEntryBase)entry);
//						if(base.created != null) {
//							System.err.println("created:");
//							base.created.printStackTrace();
//						}
//						if(base.performed != null) {
//							System.err.println("performed:");
//							base.performed.printStackTrace();
//						}
//						if(base.ready != null) {
//							System.err.println("ready:");
//							base.ready.printStackTrace();
//						}
						new Exception("Timeout waiting for request to complete: " + entry.getOriginalRequest()).printStackTrace();
						throw new DatabaseException("Timeout waiting for request to complete." +  entry.getOriginalRequest());
						//System.err.println("asd");
						//base.getQuery().recompute(null, null, entry);
					}
				}
			} catch (InterruptedException e) {
			}
		}

	}

	//////////////////////////////////////

	public static void entriesObjects(QueryProcessor processor, int r1, Consumer<Objects> consumer) {
		processor.cache.objectsMap.values(r1, consumer);
	}

	public static void entriesObjects(QueryProcessor processor, Consumer<Objects> consumer) {
		processor.cache.objectsMap.values(consumer);
	}

	public static void entriesDirectPredicates(QueryProcessor processor, Consumer<DirectPredicates> consumer) {
		processor.cache.directPredicatesMap.values(consumer);
	}

	static final void entriesDirectObjects(final QueryProcessor processor, final int r1, Consumer<DirectObjects> consumer) {
		processor.cache.directObjectsMap.values(r1, consumer);
	}

	static final void entriesStatements(final QueryProcessor processor, final int r1, Consumer<Statements> consumer) {
		processor.cache.statementsMap.values(r1, consumer);
	}

	static final Types entryTypes(final QueryProcessor processor, final int r) {
		return (Types)processor.cache.typesMap.get(r);
	}

	static final PrincipalTypes entryPrincipalTypes(final QueryProcessor processor, final int r) {
		return (PrincipalTypes)processor.cache.principalTypesMap.get(r);
	}

	static final OrderedSet entryOrderedSet(final QueryProcessor processor, final int r) {
		return (OrderedSet)processor.cache.orderedSetMap.get(r);
	}

	static final ValueQuery entryValueQuery(final QueryProcessor processor, final int r) {
		return (ValueQuery)processor.cache.valueQueryMap.get(r);
	}

	static final DirectPredicates entryDirectPredicates(final QueryProcessor processor, final int r) {
		return (DirectPredicates)processor.cache.directPredicatesMap.get(r);
	}

	public static final ReadEntry entryRead(final QueryProcessor processor, final Read request) {
		return (ReadEntry)processor.cache.readEntryMap.get(request);
	}

	public static final MultiReadEntry entryMultiRead(final QueryProcessor processor, final MultiRead request) {
		return (MultiReadEntry)processor.cache.multiReadEntryMap.get(request);
	}

	public static final AsyncReadEntry entryAsyncRead(final QueryProcessor processor, final AsyncRead request) {
		return (AsyncReadEntry)processor.cache.asyncReadEntryMap.get(request);
	}

	public static final AsyncMultiReadEntry entryAsyncMultiRead(final QueryProcessor processor, final AsyncMultiRead request) {
		return (AsyncMultiReadEntry)processor.cache.asyncMultiReadEntryMap.get(request);
	}

	protected static final long keyR2(long r1, long r2) {
		long result = (r1<<32) | (r2 & 0xffffffffL); 
		return result;
	}

	protected static final <T> T id(T o) {
		return o;
	}

	protected static final int keyR(int r) {
		return r;
	}

	protected static final String keyID(String id) {
		return id;
	}

	protected static InternalProcedure<IntSet> emptyIntSetProcedure = new InternalProcedure<IntSet>() {

		@Override
		public void execute(ReadGraphImpl graph, IntSet result) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	}; 

	protected static InternalProcedure<byte[]> emptyBytesProcedure = new InternalProcedure<byte[]>() {

		@Override
		public void execute(ReadGraphImpl graph, byte[] bytes) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	}; 

	protected static InternalProcedure<Integer> emptyIntegerProcedure = new InternalProcedure<Integer>() {

		@Override
		public void execute(ReadGraphImpl graph, Integer i) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	}; 


	protected static InternalProcedure<TObjectIntHashMap<String>> emptyNamespaceProcedure = new InternalProcedure<TObjectIntHashMap<String>>() {

		@Override
		public void execute(ReadGraphImpl graph, TObjectIntHashMap<String> i) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	}; 


	protected static InternalProcedure<RelationInfo> emptyRelationInfoProcedure = new InternalProcedure<RelationInfo>() {

		@Override
		public void execute(ReadGraphImpl graph, RelationInfo i) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	};

	protected static InternalProcedure<ObjectResourceIdMap<String>> emptyChildMapProcedure = new InternalProcedure<ObjectResourceIdMap<String>>() {

		@Override
		public void execute(ReadGraphImpl graph, ObjectResourceIdMap<String> i) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	};

	protected static IntProcedure emptyIntProcedure = new IntProcedure() {

		@Override
		public void finished(ReadGraphImpl graph) {
		}

		@Override
		public void execute(ReadGraphImpl graph, int i) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}
	};

	protected static TripleIntProcedure emptyTripleIntProcedure = new TripleIntProcedure() {

		@Override
		public void execute(ReadGraphImpl graph, int s, int p, int o) {
		}

		@Override
		public void finished(ReadGraphImpl graph) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	}; 

	protected static AsyncProcedure<Object> emptyAsyncProcedure = new AsyncProcedure<Object>() {

		@Override
		public void execute(AsyncReadGraph graph, Object result) {
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable throwable) {
		}

	};

	protected static AsyncMultiProcedure<Object> emptyAsyncMultiProcedure = new AsyncMultiProcedure<Object>() {

		@Override
		public void execute(AsyncReadGraph graph, Object result) {
		}

		@Override
		public void finished(AsyncReadGraph graph) {
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable throwable) {
		}

	}; 

	protected static SyncMultiProcedure<Object> emptySyncMultiProcedure = new SyncMultiProcedure<Object>() {

		@Override
		public void execute(ReadGraph graph, Object result) {
		}

		@Override
		public void finished(ReadGraph graph) {
		}

		@Override
		public void exception(ReadGraph graph, Throwable throwable) {
		}

	};

	protected static InternalProcedure<IntSet> emptyProcedureTypes = emptyIntSetProcedure;
	protected static InternalProcedure<IntSet> emptyProcedureSuperTypes = emptyIntSetProcedure;
	protected static InternalProcedure<IntSet> emptyProcedureTypeHierarchy = emptyIntSetProcedure;
	protected static InternalProcedure<IntSet> emptyProcedureSuperRelations = emptyIntSetProcedure;
	protected static InternalProcedure<IntSet> emptyProcedurePredicates = emptyIntSetProcedure;
	protected static InternalProcedure<IntSet> emptyProcedureDirectPredicates = emptyIntSetProcedure;

	protected static IntProcedure emptyProcedureObjects = emptyIntProcedure;
	protected static IntProcedure emptyProcedureDirectObjects = emptyIntProcedure;
	protected static IntProcedure emptyProcedurePrincipalTypes = emptyIntProcedure;
	protected static IntProcedure emptyProcedureDirectSuperRelations = emptyIntProcedure;
	protected static IntProcedure emptyProcedureAssertedPredicates = emptyIntProcedure;
	protected static IntProcedure emptyProcedureOrderedSet = emptyIntProcedure;

	protected static TripleIntProcedure emptyProcedureStatements = emptyTripleIntProcedure;
	protected static TripleIntProcedure emptyProcedureAssertedStatements = emptyTripleIntProcedure;

	protected static InternalProcedure<byte[]> emptyProcedureValueQuery = emptyBytesProcedure;

	protected static InternalProcedure<Integer> emptyProcedureURIToResource = emptyIntegerProcedure;
	protected static InternalProcedure<TObjectIntHashMap<String>> emptyProcedureNamespaceIndex = emptyNamespaceProcedure;
	protected static InternalProcedure<ObjectResourceIdMap<String>> emptyProcedureChildMap = emptyChildMapProcedure;
	protected static InternalProcedure<RelationInfo> emptyProcedureRelationInfoQuery = emptyRelationInfoProcedure;

	protected static AsyncProcedure emptyProcedureReadEntry = emptyAsyncProcedure;
	protected static AsyncProcedure emptyProcedureAsyncReadEntry = emptyAsyncProcedure;
	protected static SyncMultiProcedure emptyProcedureMultiReadEntry = emptySyncMultiProcedure;
	protected static AsyncMultiProcedure emptyProcedureAsyncMultiReadEntry = emptyAsyncMultiProcedure;
	protected static AsyncProcedure emptyProcedureExternalReadEntry = emptyAsyncProcedure;

	static class AsyncProcedureWrapper<T> implements AsyncProcedure<T> {

		private AsyncProcedure<T> procedure;
		private T result = null;
		private Throwable throwable = null;
		private Semaphore s = new Semaphore(0);

		AsyncProcedureWrapper(AsyncProcedure<T> procedure) {
			this.procedure = procedure;
		}

		@Override
		public void execute(AsyncReadGraph graph, T result) {
			if(procedure != null) procedure.execute(graph, result);
			this.result = result;
			s.release();
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable throwable) {
			if(procedure != null) procedure.exception(graph, throwable);
			this.throwable = throwable;
			s.release();
		}

		public T get() throws DatabaseException {
			try {
				s.acquire();
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
			if(throwable != null) {
				if(throwable instanceof DatabaseException) throw (DatabaseException)throwable;
				else throw new DatabaseException(throwable);
			} else {
				return result;
			}
		}
		
	}

	static class ExternalProcedureWrapper<T> implements AsyncProcedure<T> {

		private Procedure<T> procedure;
		private T result = null;
		private Throwable throwable = null;

		ExternalProcedureWrapper(Procedure<T> procedure) {
			this.procedure = procedure;
		}

		@Override
		public void execute(AsyncReadGraph graph, T result) {
			if(procedure != null) procedure.execute(result);
			this.result = result;
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable throwable) {
			if(procedure != null) procedure.exception(throwable);
			this.throwable = throwable;
		}

		public T get() throws DatabaseException {
			if(throwable != null) {
				if(throwable instanceof DatabaseException) throw (DatabaseException)throwable;
				else throw new DatabaseException(throwable);
			} else {
				return result;
			}
		}

	}


	static class InternalProcedureWrapper<T> implements InternalProcedure<T> {

		private InternalProcedure<T> procedure;
		private T result = null;
		private Throwable throwable = null;

		InternalProcedureWrapper(InternalProcedure<T> procedure) {
			this.procedure = procedure;
		}

		@Override
		public void execute(ReadGraphImpl graph, T result) throws DatabaseException {
			if(procedure != null) procedure.execute(graph, result);
			this.result = result;
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
			if(procedure != null) procedure.exception(graph, throwable);
			this.throwable = throwable;
		}

		public T get() throws DatabaseException {
			if(throwable != null) {
				if(throwable instanceof DatabaseException) throw (DatabaseException)throwable;
				else throw new DatabaseException(throwable);
			} else {
				return result;
			}
		}

	}

	static class IntSetWrapper implements IntProcedure {

		private IntProcedure procedure;
		private final IntSet result;
		private Throwable throwable = null;

		IntSetWrapper(ReadGraphImpl graph, IntProcedure procedure) {
			this.procedure = procedure;
			result = new IntSet(graph.processor.querySupport);
		}

		@Override
		public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
			if(procedure != null) procedure.execute(graph, i);
			result.add(i);
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
			if(procedure != null) procedure.exception(graph, throwable);
			this.throwable = throwable;
		}

		@Override
		public void finished(ReadGraphImpl graph) throws DatabaseException {
			if(procedure != null) procedure.finished(graph);
		}

		public IntSet get() throws DatabaseException {
			if(throwable != null) {
				if(throwable instanceof DatabaseException) throw (DatabaseException)throwable;
				else throw new DatabaseException(throwable);
			} else {
				return result;
			}
		}

	}

	static class TripleIntProcedureWrapper implements TripleIntProcedure {

		private TripleIntProcedure procedure;
		private IntArray result = new IntArray();
		private Throwable throwable = null;

		TripleIntProcedureWrapper(TripleIntProcedure procedure) {
			this.procedure = procedure;
		}

		@Override
		public void execute(ReadGraphImpl graph, int i1, int i2, int i3) throws DatabaseException {
			if(procedure != null) procedure.execute(graph, i1, i2, i3);
			result.add(i1);
			result.add(i2);
			result.add(i3);
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
			if(procedure != null) procedure.exception(graph, throwable);
			this.throwable = throwable;
		}

		@Override
		public void finished(ReadGraphImpl graph) throws DatabaseException {
			if(procedure != null) procedure.finished(graph);
		}

		public IntArray get() throws DatabaseException {
			if(throwable != null) {
				if(throwable instanceof DatabaseException) throw (DatabaseException)throwable;
				else throw new DatabaseException(throwable);
			} else {
				return result;
			}
		}

	}

	public static <T> T resultExternalReadEntry(ReadGraphImpl graph, ExternalRead r, CacheEntry parent, ListenerBase listener, Procedure<T> procedure) throws DatabaseException {
		ExternalProcedureWrapper<T> wrap = new ExternalProcedureWrapper<>(procedure);
		QueryCache.runnerExternalReadEntry(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static <T> T resultAsyncReadEntry(ReadGraphImpl graph, AsyncRead r, CacheEntry parent, ListenerBase listener, AsyncProcedure<T> procedure) throws DatabaseException {
		return (T)QueryCacheBase.runnerAsyncReadEntry(graph, r, parent, listener, procedure, true);
	}

	public static byte[] resultValueQuery(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<byte[]> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerValueQuery(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static RelationInfo resultRelationInfoQuery(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<RelationInfo> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerRelationInfoQuery(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static IntSet resultSuperRelations(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<IntSet> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerSuperRelations(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static IntSet resultSuperTypes(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<IntSet> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerSuperTypes(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static IntSet resultTypes(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<IntSet> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerTypes(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static IntSet resultPredicates(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<IntSet> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerPredicates(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static IntSet resultDirectPredicates(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<IntSet> wrap = new InternalProcedureWrapper<>(null);
		QueryCache.runnerDirectPredicates(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	public static IntArray resultAssertedStatements(ReadGraphImpl graph, int r1, int r2, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		TripleIntProcedureWrapper wrap = new TripleIntProcedureWrapper(null);
		QueryCache.runnerAssertedStatements(graph, r1, r2, parent, listener, wrap);
		return wrap.get();
	}

	public static Integer resultURIToResource(ReadGraphImpl graph, String id, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<Integer> wrap = new InternalProcedureWrapper<Integer>(null);
		QueryCache.runnerURIToResource(graph, id, parent, listener, wrap);
		return wrap.get();
	}

	public static ObjectResourceIdMap<String> resultChildMap(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener) throws DatabaseException {
		InternalProcedureWrapper<ObjectResourceIdMap<String>> wrap = new InternalProcedureWrapper<ObjectResourceIdMap<String>>(null);
		QueryCache.runnerChildMap(graph, r, parent, listener, wrap);
		return wrap.get();
	}

	static boolean shouldCache(ReadGraphImpl impl, int r) {
		return impl.processor.isImmutableForReading(r);
	}

	static boolean shouldCache(ReadGraphImpl impl, int r, int r2) {
		return impl.processor.isImmutableForReading(r);
	}

	static boolean shouldCache(ReadGraphImpl impl, ReadExt ext) throws DatabaseException {
		return ext.isImmutable(impl);
	}

	static boolean shouldCache(ReadGraphImpl impl, Object o) throws DatabaseException {
		if(o instanceof ReadExt)
			return shouldCache(impl, (ReadExt)o);
		return false;
	}

	AssertedPredicates getOrCreateAssertedPredicates(int r) {
	    AssertedPredicates entry = (AssertedPredicates)assertedPredicatesMap.get(r);
	    if(entry == null) {
	        entry = new AssertedPredicates(r);
            assertedPredicatesMap.put(keyR(r), entry);
	    }
	    return entry;
	}

    AssertedStatements getOrCreateAssertedStatements(int r1, int r2) {
        AssertedStatements entry = (AssertedStatements)assertedStatementsMap.get(r1, r2);
        if(entry == null) {
            entry = new AssertedStatements(r1, r2);
            assertedStatementsMap.put(keyR2(r1, r2), entry);
        }
        return entry;
    }

    ChildMap getOrCreateChildMap(int r) {
        ChildMap entry = (ChildMap)childMapMap.get(r);
        if(entry == null) {
            entry = new ChildMap(r);
            childMapMap.put(keyR(r), entry);
        }
        return entry;
    }

    DirectObjects getOrCreateDirectObjects(int r1, int r2) {
        DirectObjects entry = (DirectObjects)directObjectsMap.get(r1, r2);
        if(entry == null) {
            entry = new DirectObjects(r1, r2);
            directObjectsMap.put(keyR2(r1, r2), entry);
        }
        return entry;
    }
    
    DirectPredicates getOrCreateDirectPredicates(int r) {
        DirectPredicates entry = (DirectPredicates)directPredicatesMap.get(r);
        if(entry == null) {
            entry = new DirectPredicates(r);
            directPredicatesMap.put(keyR(r), entry);
        }
        return entry;
    }

    Objects getOrCreateObjects(int r1, int r2) {
        Objects entry = (Objects)objectsMap.get(r1, r2);
        if(entry == null) {
            entry = new Objects(r1, r2);
            objectsMap.put(keyR2(r1, r2), entry);
        }
        return entry;
    }

    OrderedSet getOrCreateOrderedSet(int r) {
        OrderedSet entry = (OrderedSet)orderedSetMap.get(r);
        if(entry == null) {
            entry = new OrderedSet(r);
            orderedSetMap.put(keyR(r), entry);
        }
        return entry;
    }

    Predicates getOrCreatePredicates(int r) {
        Predicates entry = (Predicates)predicatesMap.get(r);
        if(entry == null) {
            entry = new Predicates(r);
            predicatesMap.put(keyR(r), entry);
        }
        return entry;
    }

    PrincipalTypes getOrCreatePrincipalTypes(int r) {
        PrincipalTypes entry = (PrincipalTypes)principalTypesMap.get(r);
        if(entry == null) {
            entry = new PrincipalTypes(r);
            principalTypesMap.put(keyR(r), entry);
        }
        return entry;
    }

    RelationInfoQuery getOrCreateRelationInfoQuery(int r) {
        RelationInfoQuery entry = (RelationInfoQuery)relationInfoQueryMap.get(r);
        if(entry == null) {
            entry = new RelationInfoQuery(r);
            relationInfoQueryMap.put(keyR(r), entry);
        }
        return entry;
    }

    Statements getOrCreateStatements(int r1, int r2) {
        Statements entry = (Statements)statementsMap.get(r1, r2);
        if(entry == null) {
            entry = new Statements(r1, r2);
            statementsMap.put(keyR2(r1, r2), entry);
        }
        return entry;
    }

    SuperRelations getOrCreateSuperRelations(int r) {
        SuperRelations entry = (SuperRelations)superRelationsMap.get(r);
        if(entry == null) {
            entry = new SuperRelations(r);
            superRelationsMap.put(keyR(r), entry);
        }
        return entry;
    }

    SuperTypes getOrCreateSuperTypes(int r) {
        SuperTypes entry = (SuperTypes)superTypesMap.get(r);
        if(entry == null) {
            entry = new SuperTypes(r);
            superTypesMap.put(keyR(r), entry);
        }
        return entry;
    }

    TypeHierarchy getOrCreateTypeHierarchy(int r) {
        TypeHierarchy entry = (TypeHierarchy)typeHierarchyMap.get(r);
        if(entry == null) {
            entry = new TypeHierarchy(r);
            typeHierarchyMap.put(keyR(r), entry);
        }
        return entry;
    }

    Types getOrCreateTypes(int r) {
        Types entry = (Types)typesMap.get(r);
        if(entry == null) {
            entry = new Types(r);
            typesMap.put(keyR(r), entry);
        }
        return entry;
    }

    URIToResource getOrCreateURIToResource(String s) {
        URIToResource entry = (URIToResource)uRIToResourceMap.get(s);
        if(entry == null) {
            entry = new URIToResource(s);
            uRIToResourceMap.put(keyID(s), entry);
        }
        return entry;
    }

    ValueQuery getOrCreateValueQuery(int r) {
        ValueQuery entry = (ValueQuery)valueQueryMap.get(r);
        if(entry == null) {
            entry = new ValueQuery(r);
            valueQueryMap.put(keyR(r), entry);
        }
        return entry;
    }

    AsyncReadEntry getOrCreateAsyncReadEntry(ReadGraphImpl graph, AsyncRead<?> r, CacheEntry parent, ListenerBase listener, final AsyncProcedure procedure, boolean needsToBlock) throws DatabaseException {
        AsyncReadEntry existing = null;
        synchronized(asyncReadEntryMap) {
            existing = (AsyncReadEntry)asyncReadEntryMap.get(r);
            if(existing == null) {
                existing = new AsyncReadEntry(r);
                existing.setPending(querySupport);
                asyncReadEntryMap.put(id(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending(querySupport);
                return existing;
            }
        }
        if(existing.isPending()) {
            if(needsToBlock)
                waitPending(graph, existing);
            else {
                existing.executeWhenResultIsAvailable(graph.processor, new SessionTask(graph) {
                    @Override
                    public void run0(int thread) {
                        try {
                            runnerAsyncReadEntry(graph, r, parent, listener, procedure, needsToBlock);
                        } catch (DatabaseException e) {
                            LOGGER.error("Error while performing async read", e);
                        }
                    }
                });
                return null;
            }
        }
        return existing;
    }
    
    void remove(AsyncReadEntry entry) {
        synchronized(asyncReadEntryMap) {
            asyncReadEntryMap.remove(entry.id);
        }
    }
    
    public static Object runnerAsyncReadEntry(ReadGraphImpl graph, AsyncRead<?> r, CacheEntry parent, ListenerBase listener, final AsyncProcedure procedure, boolean needsToBlock) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph, r)) {
            if (SINGLE) {
                AsyncReadEntry e = cache.peekAsyncReadEntry(r);
                if (e != null && e.isReady()) {
                    return e.performFromCache(graph, procedure);
                }
            }
            return AsyncReadEntry.computeForEach(graph, r, null, procedure, needsToBlock);
        }
        AsyncReadEntry entry = (AsyncReadEntry)cache.getOrCreateAsyncReadEntry(graph, r, parent, listener, procedure, needsToBlock);
        if(entry == null) {
            // Entry was pending and this request has been queued  
            return null;
        }
        AsyncProcedure procedure_ = procedure != null ? procedure : emptyProcedureAsyncReadEntry;
        if(entry.isReady()) {
          graph.processor.listening.registerDependencies(graph, entry, parent, listener, procedure_, false);
          Object result = entry.performFromCache(graph, procedure_);
          graph.processor.listening.registerFirstKnown(listener, result);
          return result;
        }
        else {
          assert(entry.isPending());
          graph.processor.listening.registerDependencies(graph, entry, parent, listener, procedure_, false);
          Object result = AsyncReadEntry.computeForEach(graph, r, entry, procedure_, needsToBlock);
          graph.processor.listening.registerFirstKnown(listener, result);
          return result;
        }
    }
    
    AsyncReadEntry peekAsyncReadEntry(AsyncRead<?> r) {
        synchronized(asyncReadEntryMap) {
            return (AsyncReadEntry) asyncReadEntryMap.get(r);
        }
    }

}