/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.ArrayList;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.RequestFlags;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MultiReadEntry<T> extends CacheEntryBase<SyncMultiProcedure<T>> {

    private static final Logger LOGGER = LoggerFactory.getLogger(MultiReadEntry.class);

    protected MultiRead<T> id;
    
    MultiReadEntry(MultiRead<T> request) {
    	this.id = request;
    }

    @Override
    int makeHash() {
    	return id.hashCode();
    }
    
    @Override
    public Object getOriginalRequest() {
        return id;
    }
    
    @Override
    public void discard() {
    	super.discard();
    	id = null;
    	setResult(null);
    }
    
    synchronized public void finish(ReadGraph graph) {
    	
    	assert(isPending());

    	synchronized(this) {
        	setReady();
        }
        
    }

    @Override
    final public void clearResult(QuerySupport support) {
        setResult(new ArrayList<T>());
    }

    final synchronized public void addOrSet(Object item) {

    	assert(isPending());

        ArrayList<T> value = (ArrayList<T>)getResult(); 
        value.add((T)item);
        
    }
    
    @Override
    final public Query getQuery() {
    	
        return new Query() {

			@Override
			public void recompute(ReadGraphImpl graph) {
				
				try {

				    id.perform(graph , new SyncMultiProcedure<T>() {

                        @Override
                        public void execute(ReadGraph graph, T result) {
                            addOrSet(result);
                        }
                        
                        public void finished(ReadGraph graph) {
                        	finish(graph);
                        };
        				
        				@Override
        				public void exception(ReadGraph graph, Throwable t) {
                            except(t);
        	            }

                    });

				} catch (Throwable t) {
                    except(t);
                    if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                }
				
			}

			@Override
			public void removeEntry(QueryProcessor processor) {
		    	processor.cache.remove(MultiReadEntry.this);
			}

			@Override
			public int type() {
				return RequestFlags.INVALIDATE;
			}
			
			@Override
			public String toString() {
				if(id == null) return "DISCARDED";
				else return id.toString() + statusOrException;
			}
        	
        };
        
    }

    public void performFromCache(AsyncReadGraph graph, Object provider, Object procedure) {
        
        AsyncMultiProcedure<T> proc = (AsyncMultiProcedure<T>)procedure;

        if(isExcepted()) {
            
            try {
                proc.exception(graph, (Throwable)getResult());
            } catch (Throwable t) {
                LOGGER.error("performFromCache proc.exception failed", t);
            }
//            parentBarrier.dec();
            
        } else {
            
            final ArrayList<T> values = (ArrayList<T>)getResult();
            for(T value : values) {
                try {
                    proc.execute(graph, value);
                } catch (Throwable t) {
                    LOGGER.error("performFromCache proc.execute failed", t);
                }
            }

            try {
                proc.finished(graph);
            } catch (Throwable t) {
                LOGGER.error("performFromCache proc.finished failed", t);
            }
//            parentBarrier.dec();

        }
        
    }

    @Override
    public Object performFromCache(ReadGraphImpl graph, SyncMultiProcedure<T> proc) {

        if(isExcepted()) {
            
            try {
                proc.exception(graph, (Throwable)getResult());
            } catch (Throwable t) {
                LOGGER.error("performFromCache(Sync) proc.exception failed", t);
            }
            
        } else {
            
            final ArrayList<T> values = (ArrayList<T>)getResult();
            for(T value : values) {
                try {
                    proc.execute(graph, value);
                } catch (Throwable t) {
                    LOGGER.error("performFromCache(Sync) proc.execute failed", t);
                }
            }

            try {
                proc.finished(graph);
            } catch (Throwable t) {
                LOGGER.error("performFromCache(Sync) proc.finished failed", t);
            }

        }

        return null;

	}

	@Override
	public String toString() {
		if(id == null) return "DISCARDED";
		else return id.toString() + statusOrException;
	}

	public Object compute(ReadGraphImpl graph, SyncMultiProcedure<T> procedure) throws DatabaseException {
		return graph.processor.cache.performQuery(graph, id, this, procedure);
	}

}
