/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.RelationInfo;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.impl.procedure.TripleIntProcedureAdapter;
import org.simantics.db.request.RequestFlags;


final public class AssertedStatements extends CollectionBinaryQuery<TripleIntProcedure> {
    
    public AssertedStatements(final int r1, final int r2) {
        super(r1, r2);
    }

    public static AssertedStatements newInstance(final int r1, final int r2) {
        return new AssertedStatements(r1, r2);
    }

 	@Override
 	final public void removeEntry(QueryProcessor provider) {
 	    provider.cache.remove(this);
 	}
 	
 	static void computeInheritedAssertions(ReadGraphImpl graph, int type, final int predicate, final RelationInfo ri, final AssertedStatements entry, final TripleIntProcedure proc) throws DatabaseException {
 	    
        QueryProcessor processor = graph.processor;

        QueryCache.runnerDirectObjects(graph, type, processor.getInherits(), entry, null, new SyncIntProcedure() {

            @Override
            public void run(ReadGraphImpl graph) throws DatabaseException {

//                if(ri.isFunctional && found.get() == 1) {
//
//                    ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has assertions from multiple inherited types.");
//                    except(exception);
//                    proc.exception(graph, exception);
//                    return;
//                    
//                }

                finish(graph, entry);
                proc.finished(graph);
                
            }

            @Override
            public void execute(ReadGraphImpl graph, int inh) throws DatabaseException {
                
//                if(ri.isFunctional && found.get() == 1) return;

                inc();
                
                QueryCache.runnerAssertedStatements(graph, inh, predicate, entry, null, new TripleIntProcedureAdapter() {

                    @Override
                    public void execute(ReadGraphImpl graph, int s, int p, int o) throws DatabaseException {

//                        if(ri.isFunctional) {
//                            
////                            if(found.get() == 1) return;
//
//                            if(found.compareAndSet(0, o)) {
                                if(addOrSet(s,p,o, entry))
                                	proc.execute(graph, s, p, o);
//                            }
//                            // If this was a duplicate, we can ignore this
//                            else if (found.compareAndSet(o, o)) {
//                                //System.err.println("duplicates!");
//                            }
//                            // Otherwise we have error
//                            else {
//                                found.set(1);
//                            }
//                            
//                        } else {
//                            
//                            addOrSet(s, p, o);
//                            proc.execute(graph, s, p, o);
//                            
//                        }
                        
                    }

                    @Override
                    public void finished(ReadGraphImpl graph) throws DatabaseException {
                        dec(graph);
                    }
        			
        			@Override
        			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
        				proc.exception(graph, t);
        				dec(graph);
                    }
                    
                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) throws DatabaseException {
                dec(graph);
            }
            
        });

 	}

    static void computeLocalAssertions(ReadGraphImpl graph, final int type, final int predicate, final RelationInfo ri, final AssertedStatements entry, final TripleIntProcedure proc) throws DatabaseException {
        
        
    }

	//@Override
	public Object compute(ReadGraphImpl graph, final TripleIntProcedure procedure) throws DatabaseException {
		computeForEach(graph, r1(), r2(), this, procedure);
		return getResult();
	}

    public static void computeForEach(ReadGraphImpl graph, final int type, final int predicate, final AssertedStatements entry, final TripleIntProcedure procedure) throws DatabaseException {

    	RelationInfo ri = QueryCache.resultRelationInfoQuery(graph, predicate, entry, null);
    	
        final AtomicInteger found = new AtomicInteger(0);

        QueryProcessor processor = graph.processor;
        
        QueryCache.runnerDirectObjects(graph, type, processor.getAsserts(), entry, null, new SyncIntProcedure() {

            @Override
            public void run(ReadGraphImpl graph) throws DatabaseException {
                
                if(ri.isFunctional && found.get() > 1) {

                    ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one asserted statement.");
                    except(exception, entry);
                    procedure.exception(graph, exception);
                    return;
                    
                }

                if(ri.isFunctional && found.get() == 1) {
                                        
                    finish(graph, entry);
                    procedure.finished(graph);
                    return;
                    
                }

                computeInheritedAssertions(graph, type, predicate, ri, entry, procedure);
                
            }

            @Override
            public void execute(ReadGraphImpl graph, final int ass) throws DatabaseException {
                
                if(ri.isFunctional && found.get() > 1) return;
                
                inc();
                
                QueryCache.runnerDirectObjects(graph, ass, processor.getHasPredicate(), entry, null, new IntProcedure() {

                    @Override
                    public void execute(ReadGraphImpl graph, final int pred) throws DatabaseException {
                        
                        if(ri.isFunctional) {
                            
                            if(found.get() > 1) return;

                            inc();

                            QueryCache.runnerDirectObjects(graph, ass, processor.getHasObject(), entry, null, new IntProcedure() {

                                @Override
                                public void execute(ReadGraphImpl graph, final int object) throws DatabaseException {
                                    
                                    if(found.get() > 1) return;

                                    if(pred == predicate) {

                                    	if(found.getAndIncrement() == 0) {
                                    		if(addOrSet(type, pred, object, entry))
                                    			procedure.execute(graph, type, pred, object);
                                    	}
                                    	
                                        return;
                                        
                                    }
                                    
                                    if ( !ri.isFinal ) {

                                        inc();

                                        QueryCache.runnerSuperRelations(graph, pred, entry, null, new InternalProcedure<IntSet>() {
                                            
                                            @Override
                                            public void execute(ReadGraphImpl graph, IntSet result) throws DatabaseException {
                                                
                                                if(found.get() > 1) {
                                                    dec(graph);
                                                    return;
                                                }

                                                if(result.contains(predicate)) {
                    
                                                    if(found.getAndIncrement() == 0) {
	                                                	if(addOrSet(type, pred, object, entry))
	                                                		procedure.execute(graph, type, pred, object);
                                                	}
                                                    
                                                }
                                                
                                                dec(graph);
                                                
                                            }
                                			
                                			@Override
                                			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                			    
                                				procedure.exception(graph, t);
                                                dec(graph);
                                				
                                            }

                                        });

                                    }

                                }
                                
                                @Override
                                public void finished(ReadGraphImpl graph) throws DatabaseException {
                                    dec(graph);
                                }
                    			
                    			@Override
                    			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                    				procedure.exception(graph, t);
                                    dec(graph);
                                }

                            });
                            
                        } else {

                            inc();

                            QueryCache.runnerDirectObjects(graph, ass, processor.getHasObject(), entry, null, new IntProcedure() {

                                @Override
                                public void execute(ReadGraphImpl graph, final int object) throws DatabaseException {
                                    
                                    if(pred == predicate) {

                                        addOrSet(type, pred, object, entry);
                                        procedure.execute(graph, type, pred, object);
                                        return;
                                        
                                    }
                                    
                                    if ( !ri.isFinal ) {

                                        inc();

                                        QueryCache.runnerSuperRelations(graph, pred, entry, null, new InternalProcedure<IntSet>() {
                                            
                                            @Override
                                            public void execute(ReadGraphImpl graph, IntSet result) throws DatabaseException {
                                                
                                                if(result.contains(predicate)) {
                    
                                                    addOrSet(type, pred, object, entry);
                                                    procedure.execute(graph, type, pred, object);

                                                }
                                                
                                                dec(graph);
                                                
                                            }
                                			
                                			@Override
                                			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                				procedure.exception(graph, t);
                                                dec(graph);
                                            }
 
                                        });

                                    }

                                }
                                
                                @Override
                                public void finished(ReadGraphImpl graph) throws DatabaseException {
                                    dec(graph);
                                }
                    			
                    			@Override
                    			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                    				procedure.exception(graph, t);
                                    dec(graph);
                                }

                            });

                        }

                    }

                    @Override
                    public void finished(ReadGraphImpl graph) throws DatabaseException {
                        dec(graph);
                    }
        			
        			@Override
        			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
        				procedure.exception(graph, t);
                        dec(graph);
                    }

                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) throws DatabaseException {
                dec(graph);
            }
            
        });
        
    }
    
    @Override
    public String toString() {
    	return "AssertedStatements[" + r1() + " - " + r2() + "]";
    }

    private boolean addOrSet(int s, int p, int o) {
    	
    	assert(isPending());
        
        IntArray value = (IntArray)getResult();
        if(value.data != null) {
			for(int i = 0;i < value.sizeOrData ; i+=3) {
				int existingP = value.data[i+1];
				if(p == existingP) {
					int existingO = value.data[i+2];
					if(existingO == o) return false;
				}
			}
        	value.add(s);
        	value.add(p);
        	value.add(o);
        } else {
        	value.add(s);
        	value.add(p);
        	value.add(o);
        }
        
        return true;
        
    }
    
    static boolean addOrSet(int s, int p, int o, AssertedStatements entry) {
    	if(entry != null) {
    		return entry.addOrSet(s, p, o);
    	} else {
    		return true;
    	}
    }
    
    static void finish(ReadGraphImpl graph, AssertedStatements entry) {
        
    	assert(entry.isPending());
    	if(entry != null) {
    		synchronized(entry) {
    			entry.setReady();
    		}
    	}

    }
    
    static void except(Throwable t, AssertedStatements entry) {
    	if(entry != null) {
    		synchronized(entry) {
    			entry.except(t);
    		}
    	}
    }

    @Override
    public Object performFromCache(ReadGraphImpl graph, final TripleIntProcedure procedure) throws DatabaseException {
        
    	assert(isReady());
        
    	if(handleException(graph, procedure)) return getResult();
    	
        final IntArray value = (IntArray)getResult();
        for(int i=0;i<value.size();i+=3) {
        	procedure.execute(graph, value.data[i], value.data[i+1], value.data[i+2]);
        }

        procedure.finished(graph);
        
        return value;
        
    }
    
    @Override
    public void recompute(ReadGraphImpl graph) throws DatabaseException {
        
        compute(graph, new TripleIntProcedureAdapter() {

            @Override
            public void finished(ReadGraphImpl graph) {
            }
			
			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				throw new Error("Error in recompute", t);
            }
    
        });
        
    }
    
    @Override
    public int type() {
        return RequestFlags.IMMEDIATE_UPDATE;
    }
    

    @Override
    boolean isImmutable(ReadGraphImpl graph) {
        return graph.processor.isImmutableForReading(r1());
    }

}
