/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.compressions;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;

/**
 * A facade for accessing {@link CompressionCodec} implementations.
 * 
 * @author Tuukka Lehtonen
 */
public final class Compressions {

	private static final Map<String, CompressionCodec> CODECS = new HashMap<>();
    private static boolean initialized = false;

    public static final String LZ4 = "LZ4";
    public static final String FASTLZ = "FASTLZ";
    
    private static void collectCodecs() {
        try {
            Collection<ServiceReference<CompressionCodec>> serv = Activator.getContext().getServiceReferences(CompressionCodec.class, null);
            for (ServiceReference<CompressionCodec> service : serv) {
                CompressionCodec codec = Activator.getContext().getService(service);
                CODECS.put(codec.getId(), codec);
            }
            initialized = true;
        } catch (InvalidSyntaxException e) {
            e.printStackTrace();
        }
    }
    
	
	/**
	 * Get a {@link CompressionCodec} matching the requested name.
	 * @param <T>
	 * 
	 * @param codec codec name
	 * @return 
	 * @return matching codec
	 * @see #FASTLZ
	 * @see #LZ4
	 */
	public static CompressionCodec get(String codecId) {
	    if (!initialized) {
	        collectCodecs();
	    }
	    CompressionCodec codec = CODECS.get(codecId);
	    if (codec == null) {
	        throw new IllegalArgumentException("No codec with id " + codecId + " installed");
	    }
	    return codec;
	}
	
	/**
	 * @param codecId
	 * @param file
	 * @return
	 * @throws FileNotFoundException 
	 */
	public static OutputStream write(String codecId, File file) throws FileNotFoundException {
		return get(codecId).write(file);
	}

	/**
	 * @param codecId
	 * @param file
	 * @return
	 * @throws FileNotFoundException 
	 */
	public static InputStream read(String codecId, File file) throws FileNotFoundException {
		return get(codecId).read(file);
	}
}