package org.simantics.charts;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.simantics.charts.editor.ChartData;
import org.simantics.charts.editor.ChartKeys;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.util.Bean;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.history.HistoryException;
import org.simantics.history.HistorySamplerItem;
import org.simantics.history.HistorySamplerItem2;
import org.simantics.history.ItemManager;
import org.simantics.history.HistorySamplerItem2.LevelItem;
import org.simantics.history.util.subscription.SamplingFormat;
import org.simantics.modeling.subscription.SubscriptionItem;
import org.simantics.modeling.subscription.SubscriptionItemQuery;
import org.simantics.project.IProject;
import org.simantics.simulation.experiment.IExperiment;

/**
 * Main facade for externally dealing with the trending system.
 * 
 * @author Tuukka Lehtonen
 * @author Antti Villberg
 * 
 */
public final class Charts {

    public static void resetChartEditorData(IProject project, Resource model, ChartData editorData) {
        if (editorData != null) {
            project.setHint(ChartKeys.chartSourceKey(model), editorData);
        } else {
            project.removeHint(ChartKeys.chartSourceKey(model));
        }
    }

	public static HistorySamplerItem createHistorySamplerItem(ReadGraph graph, Variable run, Resource subscriptionItem) throws DatabaseException {
			IExperiment exp = (IExperiment) run.getPropertyValue(graph, "iExperiment");
			ITrendSupport support = exp.getService(ITrendSupport.class);
			ChartData data = support.getChartData();
			return createHistorySamplerItem(graph, subscriptionItem, data);
	}

    public static HistorySamplerItem createHistorySamplerItem(ReadGraph graph, Resource subscriptionItem, ChartData data) throws DatabaseException {

        try {
            Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(subscriptionItem));
            if (indexRoot == null) {
                throw new DatabaseException("There is no index root for " + subscriptionItem);
            }

            if(data == null) {
                throw new DatabaseException("There is no chart data for " + subscriptionItem);
            }

            ItemManager im = new ItemManager(data.history.getItems());

            SubscriptionItem i = graph.syncRequest(new SubscriptionItemQuery(subscriptionItem));

            List<Bean> items = im.search("variableId", i.variableId);
            Collections.sort(items, SamplingFormat.INTERVAL_COMPARATOR);
            if (items.isEmpty())
                throw new DatabaseException("There is history item for " + i.variableId);
            Bean config = items.get(0);
            String historyId = (String) config.getFieldUnchecked("id");

            return new HistorySamplerItem(data.collector, data.history, historyId, System.identityHashCode(data));
        } catch (HistoryException e) {
            throw new DatabaseException(e);
        } catch (BindingException e) {
            throw new DatabaseException(e);
        }
    }

    public static HistorySamplerItem2 createHistorySamplerItem2(ReadGraph graph, Variable run, Resource subscriptionItem) throws DatabaseException {
        IExperiment exp = (IExperiment) run.getPossiblePropertyValue(graph, "iExperiment");
        if (exp == null)
            return null;
        ITrendSupport support = exp.getService(ITrendSupport.class);
        ChartData data = support.getChartData();
        return data != null ? createHistorySamplerItem2(graph, subscriptionItem, data) : null;
    }

    public static HistorySamplerItem2 createHistorySamplerItem2(ReadGraph graph, Resource subscriptionItem, ChartData data) throws DatabaseException {
        try {
            Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(subscriptionItem));
            if (indexRoot == null)
                throw new DatabaseException("There is no index root for " + subscriptionItem);
            if (data == null)
                throw new DatabaseException("There is no chart data for " + subscriptionItem);

            SubscriptionItem i = graph.syncRequest(new SubscriptionItemQuery(subscriptionItem));

            // TODO: this is super-inefficient!
            ItemManager im = new ItemManager(data.history.getItems());

            List<Bean> items = im.search("variableId", i.variableId);
            if (items.isEmpty())
                throw new DatabaseException("There is history item for " + i.variableId);

            LevelItem[] historyItems = items.stream().map(Charts::toLevelItem).sorted(ASCENDING_INTERVAL_ORDER).toArray(LevelItem[]::new);

            return new HistorySamplerItem2(data.collector, data.history, historyItems, System.identityHashCode(data));
        } catch (HistoryException e) {
            throw new DatabaseException(e);
        } catch (BindingException e) {
            throw new DatabaseException(e);
        }
    }

    private static LevelItem toLevelItem(Bean bean) {
        String id = (String) bean.getFieldUnchecked("id");
        double interval = (Double) bean.getFieldUnchecked("interval");
        return new LevelItem(id, interval);
    }

    public final static Comparator<LevelItem> ASCENDING_INTERVAL_ORDER =
            (o1, o2) -> SamplingFormat.compareSamplingInterval(o1.samplingInterval, o2.samplingInterval);

}
