/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Consumer;

import org.eclipse.core.runtime.IAdaptable;
import org.simantics.browsing.ui.NodeContext.CacheKey;
import org.simantics.browsing.ui.NodeContext.PrimitiveQueryKey;
import org.simantics.browsing.ui.NodeContext.QueryKey;
import org.simantics.browsing.ui.content.Labeler;
import org.simantics.browsing.ui.content.Labeler.Modifier;
import org.simantics.utils.threads.IThreadWorkQueue;

/**
 * 
 */
public interface GraphExplorer extends IAdaptable {
	
    /**
     * A key that can be used to associate GraphExplorer instances with other
     * objects in, e.g. SWT widgets using Widget.setData(String, Object).
     */
    public static final String KEY_GRAPH_EXPLORER = "GraphExplorer"; //$NON-NLS-1$

    /**
     * @see #setAutoExpandLevel(int)
     */
    public static int ALL_LEVELS = -1;

    public static final Object EMPTY_INPUT = new Object() {

        @Override
        public String toString() { return "GraphExplorer.EMPTY_INPUT"; }; //$NON-NLS-1$

    };

    /**
     * @deprecated not needed anymore, just remove all uses
     */
    @Deprecated
    public interface ModificationContext extends IAdaptable {

    }

    /**
     * An IGraphExplorer has a UI which uses some toolkit (AWT/SWT). The
     * toolkits are assumed to be single-threaded. The return value of this
     * method indicates the thread used for UI-related accesses.
     * 
     * @return the single access thread used by this GraphExplorer
     */
    IThreadWorkQueue getThread();

    /**
     * Override this method to customize the maximum amount of child nodes that
     * are shown for any node. The returned value must be non-negative.
     * 
     * Use {@link Integer#MAX_VALUE} to maximize the limit.
     * 
     * @return max amount of children shown for a node, must not be negative
     */
    int getMaxChildren();
    void setMaxChildren(int maxChildren);

    /**
     * Query for context-specific max children to show value using the provided
     * query manager and node context.
     * 
     * @param manager
     * @param context
     * @return node specific child limit or what is returned by
     *         {@link #getMaxChildren()} if node specific limit is not set
     */
    int getMaxChildren(NodeQueryManager manager, NodeContext context);

    /**
     * TODO: documentation
     * 
     * @param <T>
     * @param processor
     */
    <T> void setProcessor(NodeQueryProcessor<T> processor);
    <T> NodeQueryProcessor<T> getProcessor(QueryKey<T> key);

    /**
     * TODO: documentation
     * 
     * @param <T>
     * @param processor
     */
    <T> void setPrimitiveProcessor(PrimitiveQueryProcessor<T> processor);
    <T> PrimitiveQueryProcessor<T> getPrimitiveProcessor(PrimitiveQueryKey<T> key);

    /**
     * Associates the provided DataSource implementation with class T in this
     * GraphExplorer. PrimitiveQueryProcessor implementations can access these
     * data sources through the class provided by
     * {@link DataSource#getProvidedClass()}. Specifying a new association for
     * the same T will overwrite the previous data source association.
     * 
     * @param <T> the provided data source type
     * @param dataSource the data source
     */
    <T> void setDataSource(DataSource<T> dataSource);

    /**
     * Removes the data source associated with the provided class.
     * 
     * @param forProvidedClass
     * @return the removed data source association
     */
    <T> DataSource<T> removeDataSource(Class<T> forProvidedClass);

    void setUIContexts(Set<String> contexts);

    /**
     * Set the root input object of this explorer.
     * @param root the input object, may <b>not</b> be <code>null</code>
     */
    void setRoot(Object root);

    /**
     * Set the root node context of this explorer. Allows the client to provide
     * the complete NodeContext to be used as the explorer root input while
     * {@link #setRoot(Object)} only allows the client to provide the
     * {@link BuiltinKeys#INPUT} data for the root NodeContext.
     * 
     * @param root the root node context, may <b>not</b> be <code>null</code>
     */
    void setRootContext(NodeContext context);

    /**
     * Get the current UI item context of the root input object.
     * 
     * Return value is undefined for a disposed or uninitialized GraphExplorer.
     * An explorer is uninitialized when {@link #setRoot(Object)} or
     * {@link #setRootContext(NodeContext)} has not been invoked yet.
     * 
     * @return input object's UI item context
     */
    NodeContext getRoot();

    /**
     * Get the parent node context of the specified node context. If there is no
     * visible parent node in the explorer, <code>null</code> is returned.
     * 
     * Only to be invoked from the UI thread (see {@link #getThread()}.
     * 
     * @param context the context to the possible parent node for
     * @return <code>null</code> if there is no parent node
     */
    NodeContext getParentContext(NodeContext context);

    /**
     * Consult the explorer's cache for the specified node and key. The method
     * will return a result only if the requested query has been previously
     * performed and has not been cleared (invalidated).
     * 
     * @param <T>
     * @param context the UI item's context
     * @param key the cache key to consult for
     * @return the value in the cache or <code>null</code> if no value was found
     */
    <T> T query(NodeContext context, CacheKey<T> key);

    // --- UI related things ---

    public interface TransientExplorerState {
    	Integer getActiveColumn();	
    }
    
    Object getWidgetSelection();
    TransientExplorerState getTransientState();
    
    SelectionDataResolver getSelectionDataResolver();
    void setSelectionDataResolver(SelectionDataResolver r);

    /**
     * @return the current {@link SelectionFilter}
     */
    SelectionFilter getSelectionFilter();

    /**
     * Set new {@link SelectionFilter} for this explorer to control what will be
     * put into selections provided by this <code>GraphExplorer</code>.
     * 
     * TODO: specify what should happen to the current provided selection when a
     * new filter is set. Should it change immediately or be left as is?
     * 
     * @param f the new filter or <code>null</code> to disable filtering
     */
    void setSelectionFilter(SelectionFilter f);

    void setSelectionTransformation(BiFunction<GraphExplorer, Object[], Object[]> f);

    //ISelectionProvider getSelectionProvider();

    void setColumnsVisible(boolean visible);
    void setColumns(Column[] columns);
    void setColumns(Column[] columns, Consumer<Map<Column, Object>> callback);
    Column[] getColumns();

    /**
     * @param context
     */
    boolean select(NodeContext context);
    boolean selectPath(Collection<NodeContext> contexts);
    boolean isVisible(NodeContext context);

    /**
     * Sets the expanded state of the specified node context.
     * 
     * <p>
     * May work asynchronously.
     * </p>
     * 
     * @param context the context to set the expanded state for
     * @param expanded <code>true</code> to expand, <code>false</code> to
     *        collapse
     */
    void setExpanded(NodeContext context, boolean expanded);

    <T> void addListener(T listener);
    <T> void removeListener(T listener);

    void setFocus();

    /**
     * Attempts to start in-line editing at the specified INodeContext in this
     * graph explorer. The nature of the in-line editing (i.e. used controls)
     * depends on the kind of {@link Modifier} returned by the {@link Labeler}
     * of the specified node context. The labeler is determined by the
     * {@link BuiltinKeys#SELECTED_LABELER} query.
     * 
     * <p>
     * Only to be invoked from the UI thread (see {@link #getThread()}.
     * 
     * @param context the node, i.e. tree item or tree-table row to start
     *        editing on
     * @param columnKey a key to identify the column to edit. This argument must
     *        match one of the keys of the tree columns set using
     *        {@link #setColumns(Column[])}.
     * @return <code>true</code> if the editing process was successfully
     *         initiated
     */
    String startEditing(NodeContext context, String columnKey);
    String startEditing(String columnKey);

    /**
     * @return <code>true</code> if the explorer is disposed.
     */
    boolean isDisposed();

    /**
     * Return the backing UI control of this GraphExplorer.
     * 
     * @param <T>
     * @return
     */
    <T> T getControl();

    /**
     * @deprecated not needed anymore, just remove calls
     */
    @Deprecated
    void setModificationContext(ModificationContext modificationContext);

    /**
     * Sets the auto-expand level to be used when the input of the viewer is set
     * using {@link #setInput(Object)}. The value 0 means that there is no
     * auto-expand; 1 means that the invisible root element is expanded (since
     * most concrete subclasses do not show the root element, there is usually
     * no practical difference between using the values 0 and 1); 2 means that
     * top-level elements are expanded, but not their children; 3 means that
     * top-level elements are expanded, and their children, but not
     * grandchildren; and so on.
     * <p>
     * The value {@value #ALL_LEVELS} means that all subtrees should be
     * expanded.
     * </p>
     * 
     * @param level non-negative level, or {@value #ALL_LEVELS} to expand all
     *        levels of the tree
     */
    void setAutoExpandLevel(int level);

    /**
     * Sets the implementation that this GraphExplorer shall use to persist its
     * state when disposed and to load its state when created.
     * 
     * In order to restore a previously persisted state, this method must be
     * called before invoking {@link #setRoot(Object)} or
     * {@link #setRootContext(NodeContext)}.
     * 
     * @param persistor the persistor to use
     */
    void setPersistor(StatePersistor persistor);

    /**
     * @return <code>true</code> if the explorer is in editable state
     * @see #setEditable(boolean)
     */
    boolean isEditable();

    /**
     * Makes this explorer editable or non-editable as far as this
     * implementation is concerned. For a non-editable explorer
     * {@link #startEditing(String)} and
     * {@link #startEditing(NodeContext, String)} will do nothing. Externally
     * added listeners (see {@link #addListener(Object)}) will still get
     * notified of events so those need to be handled separately.
     * 
     * <p>
     * By default, implementations should be editable.
     */
    void setEditable(boolean editable);
    
    /**
     * Returns underlaying data object, which was clicked (with mouse).
     * @param event Mouse Event (usually org.eclipse.swt.events.MouseEvent) 
     */
    public Object getClicked(Object event);

}
