/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.extension;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.browsing.ui.common.Activator;
import org.simantics.browsing.ui.common.extension.internal.EvaluatorBindingExtension;
import org.simantics.browsing.ui.common.extension.internal.EvaluatorImplementationBinding;
import org.simantics.browsing.ui.common.extension.internal.EvaluatorReferenceBinding;
import org.simantics.utils.strings.StringUtils;

/**
 * @author Tuukka Lehtonen
 */
public class EvaluatorBindingExtensionManager implements IExtensionChangeHandler {

    private final static String                NAMESPACE    = Activator.PLUGIN_ID;

    private final static String                EP_NAME      = "evaluatorBinding";

    private ExtensionTracker                   tracker;

    private EvaluatorBindingExtension[]            extensions   = new EvaluatorBindingExtension[0];

    private static EvaluatorBindingExtensionManager INSTANCE;


    public static synchronized EvaluatorBindingExtensionManager getInstance() {
        if (INSTANCE == null)
            INSTANCE = new EvaluatorBindingExtensionManager();
        return INSTANCE;
    }

    public static synchronized void dispose() {
        if (INSTANCE != null) {
            INSTANCE.close();
            INSTANCE = null;
        }
    }


    private EvaluatorBindingExtensionManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    private void close() {
        tracker.close();
        tracker = null;
        extensions = new EvaluatorBindingExtension[0];
    }

    public EvaluatorBindingExtension[] getExtensions() {
        return extensions;
    }

    private void loadExtensions(IConfigurationElement[] elements) {

        Set<EvaluatorBindingExtension> newExtensions = new HashSet<EvaluatorBindingExtension>(Arrays.asList(extensions));

        for (IConfigurationElement el : elements) {

            String browseContext = StringUtils.safeString(el.getAttribute("browseContext"));

            for(IConfigurationElement child : el.getChildren("reference")) {

                String factoryId = StringUtils.safeString(child.getAttribute("id"));
                EvaluatorBindingExtension ext = new EvaluatorReferenceBinding(browseContext, factoryId);

                // Start tracking the new extension object, its removal will be notified of
                // with removeExtension(extension, Object[]).
                tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                newExtensions.add(ext);

            }

            for(IConfigurationElement child : el.getChildren("implementation")) {

                try {

                    EvaluatorFactory factory = (EvaluatorFactory)child.createExecutableExtension("class");
                    EvaluatorBindingExtension ext = new EvaluatorImplementationBinding(browseContext, factory);

                    // Start tracking the new extension object, its removal will be notified of
                    // with removeExtension(extension, Object[]).
                    tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                    newExtensions.add(ext);

                } catch (CoreException e) {

                    System.out.println(" == Could not load ViewpointContributionFactory '" + child.getAttribute("class") + "' due to the following error: " + e.getMessage()  );

                }

            }

        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new EvaluatorBindingExtension[newExtensions.size()]);
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public void removeExtension(IExtension extension, Object[] objects) {
        Set<EvaluatorBindingExtension> newExtensions = new HashSet<EvaluatorBindingExtension>(Arrays.asList(extensions));

        for (Object o : objects) {
            tracker.unregisterObject(extension, o);
            newExtensions.remove(o);
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new EvaluatorBindingExtension[newExtensions.size()]);
    }

    public Collection<EvaluatorFactory> getBoundFactories(Set<String> browseContexts) {

//        ArrayList<EvaluatorFactory> result = new ArrayList<EvaluatorFactory>();
//        for(EvaluatorBindingExtension binding : getExtensions()) {
//            if(browseContexts.contains(binding.browserId())) {
//                EvaluatorFactory factory = EvaluatorFactoryExtensionManager.getFactory(binding.factoryId());
//                result.add(factory);
//            }
//        }
//        return result;

        HashSet<EvaluatorFactory> result = new HashSet<EvaluatorFactory>();

        for(EvaluatorBindingExtension binding : getExtensions()) {
            if(browseContexts.contains(binding.getBrowseContext())) {
                EvaluatorFactory factory = binding.getFactory();
                if(factory != null) {
//                    System.out.println("----------- Plugin contribution " + binding.getFactory());
                    result.add(factory);
                } else {
//                    System.out.println("FAILED: ----------- No factory for " + binding);
                }

            }
        }

        return result;

    }

}
