/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common;

import java.util.Collection;

import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.Tester;
import org.simantics.browsing.ui.content.CheckedStateFactory;
import org.simantics.browsing.ui.content.ComparableContextFactory;
import org.simantics.browsing.ui.content.ImageDecoratorFactory;
import org.simantics.browsing.ui.content.ImagerFactory;
import org.simantics.browsing.ui.content.LabelDecoratorFactory;
import org.simantics.browsing.ui.content.LabelerFactory;
import org.simantics.browsing.ui.content.ViewpointFactory;
import org.simantics.utils.datastructures.ComparatorFactory;

/**
 * An interface for retrieving a collection of {@link Evaluator}s based on any
 * input object ({@link #get(Object)}.
 * 
 * <p>
 * {@link EvaluatorDataImpl} is the default class-based concrete implementation of
 * this interface that clients can and should use.
 * </p>
 * 
 * @see Evaluator
 * @see EvaluatorDataImpl
 */
public interface EvaluatorData {

    /**
     * A transformer from one input object to another. The idea is that before
     * resolving an Evaluator for an input object, all possible input
     * transformations have been executed.
     * 
     * <p>
     * Consider classes A and B which have are of a separate inheritance
     * hierarchy and share no common interfaces. In principle, this mechanism
     * should allow clients to define evaluators for class B, define a
     * transformer from class A to B and give input objects of class A, and
     * evaluator for class B would get resolved for these inputs.
     * </p>
     * 
     * NOTE: this mechanism is not in use yet.
     * 
     */
    public interface Transformer {
        Object transform(Object source);
    }

    /**
     * An evaluator describes available ways of
     * <ul>
     * <li>producing content for a {@link GraphExplorer}, i.e. detemining the
     * children of a {@link NodeContext} (see {@link ViewpointFactory})</li>
     * <li>sorting the produced child content for a {@link NodeContext} (see
     * {@link ComparableContextFactory})</li>
     * <li>producing labels for the produced {@link NodeContext}s (see
     * {@link LabelerFactory})</li>
     * <li>decorating labels produced for {@link NodeContext}s (see
     * {@link LabelDecoratorFactory})</li>
     * </ul>
     * 
     * An evaluator may produce several alternatives for each of the above
     * tasks, sorted by preference numbers bound to the factories. Picking the
     * factory to use is not the task of the evaluator.
     * 
     * @see EvaluatorImpl
     */
    public interface Evaluator {

        EvaluatorTree<ComparableContextFactory> getComparableContextTree();
        EvaluatorTree<ImagerFactory> getImagerTree();
        EvaluatorTree<ImageDecoratorFactory> getImageDecoratorTree();
        EvaluatorTree<LabelDecoratorFactory> getLabelDecoratorTree();
        EvaluatorTree<LabelerFactory> getLabelerTree();
        EvaluatorTree<CheckedStateFactory> getCheckStateTree();
        EvaluatorTree<ViewpointFactory> getViewpointTree();

        /**
         * A convenience method for building an evaluator by adding viewpoint to
         * the ViewpointFactory EvaluatorTree root. Maybe should not be in the
         * interface.
         * 
         * @param factory the factory to add
         * @param preference
         * @return this evaluator to allow chained initialization
         */
        Evaluator addViewpoint(ViewpointFactory factory, double preference);
        Evaluator addComparableContext(ComparableContextFactory factory, double preference);
        Evaluator addLabeler(LabelerFactory factory, double preference);
        Evaluator addCheckState(CheckedStateFactory factory, double preference);
        Evaluator addLabelDecorator(LabelDecoratorFactory factory, double preference);
        Evaluator addComparator(ComparableContextFactory factory, double preference);
        Evaluator addImager(ImagerFactory factory, double preference);
        Evaluator addImageDecorator(ImageDecoratorFactory factory, double preference);

    };

    /**
     * A node in a tree of factories. Each node is associated with a
     * {@link Tester} that determines whether the factories associated with this
     * node should be included in the available factory evaluation result of the
     * specified {@link NodeContext} input.
     * 
     * <p>
     * The factory tree structure allows optimization of factory resolution when
     * there are lots of factories available. By grouping factories under tree
     * nodes with a suitable tester allows more efficient result calculation.
     * 
     * <p>
     * Should not be created directly. Use
     * <code>Evaluator.add*Branch(Tester)</code> to create new evaluation tree
     * branches.
     * 
     * @param <Factory> one of factories listed as "see also"
     * 
     * @see ViewpointFactory
     * @see ComparatorFactory
     * @see LabelerFactory
     * @see LabelDecoratorFactory
     */
    public interface EvaluatorTree<Factory> {

        /**
         * @return a tester to indicate whether the factories in this tree node
         *         should be added to the factory evaluation result.
         */
        Tester getTester();

        /**
         * @param factory
         * @param preference
         * @return this {@link EvaluatorTree} instance for chained initialization
         */
        EvaluatorTree<Factory> addFactory(Factory factory, double preference);

        /**
         * @param tester
         * @return the new {@link EvaluatorTree} branch created as a child of
         *         this {@link EvaluatorTree}
         */
        EvaluatorTree<Factory> addBranch(Tester tester);

        /**
         * @return preference-wrapped factories accepted by this EvaluatorTree node
         */
        Collection<Preference<Factory>> getAcceptedFactories();

        /**
         * @return preference-wrapped child tree nodes of this EvaluatorTree node
         */
        Collection<EvaluatorTree<Factory>> getChildren();

    }

    /**
     * Creates a new Evaluator that is completely separate from this
     * {@link EvaluatorData}. {@link #addEvaluator(Class, Evaluator)} can later
     * be used to bind the evaluator to any number of EvaluatorData instances.
     * 
     * @return new {@link Evaluator} instance
     */
    Evaluator newEvaluator();

    /**
     * Bind the specified evaluator to the specified class. After this results
     * of {@link #get(Object)} should include <code>eval</code> for any input
     * object that is an instance of <code>clazz</code>.
     * 
     * @param clazz
     * @param eval
     */
    void addEvaluator(Class<?> clazz, Evaluator eval);

    /**
     * Attempts to get a collection of possible <code>Evaluator</code> instances
     * for the specified input object.
     * 
     * <p>
     * The resolution of "possible evaluators" can be based on, e.g. Class
     * comparison, <code>instanceof</code> queries or adaptation through
     * Eclipse's <code>IAdaptable</code> interface.
     * </p>
     * 
     * @param input any object that represents the input for producing some
     *        output.
     * @return an empty collection if no <code>Evaluator</code> could be found
     *         to take care of the specified kind of input
     */
    Collection<Evaluator> get(Object input);

    /**
     * An evaluator entry descriptor returned by
     * {@link EvaluatorData#enumEvaluators()}.
     */
    public static interface EvaluatorEntry {
        public Class<?> getClazz();
        public Collection<Evaluator> getEvaluators();
    };

    /**
     * Enumerates all the evaluators added by the client using
     * {@link #addEvaluator(Class, Evaluator)}.
     */
    Collection<EvaluatorEntry> enumEvaluators();
    <T> T getBrowseContext();

}
