/*******************************************************************************
 * Copyright (c) 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.acorn;

import java.io.IOException;
import java.util.Objects;

import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.simantics.db.IO;

/**
 * @author Antti Villberg
 */
public class AcornKeyRocks implements AcornKey {

	private RocksStore store;
	private AcornKeyRocks parent;
	private String name;

	public AcornKeyRocks(RocksStore store, AcornKeyRocks parent, String name) {
		this.store = store;
		this.parent = parent;
		this.name = name;
	}

	public IO getIO() {
		return new RocksIO(store.db(), this);
	}

	/*
	 * $dirName1
	 * dirName1.child1
	 * $dirName2
	 * dirName2.child1
	 * dirName2.child2
	 */
	public String key() {
		if(parent != null)
			return parent.name + "." + name;
		else
			return "$" + name;
	}

	public byte[] keyBytes() {
		return key().getBytes();
	}

	public boolean exists() throws IOException {
		try {
			return getIO().length() != 0;
		} catch (IOException e) {
			return false;
		}
	}

	public String getName() {
		return name;
	}

	public String debugLocation() {
		return key();
	}

	public AcornKey child(String name) {
		return new AcornKeyRocks(store, this, name);
	}

	public void ensureExists() throws IOException {
		byte[] MAGIC = "dir".getBytes();
		getIO().saveBytes(MAGIC, MAGIC.length, true);
	}

	public boolean sameName(String n) {
		return name.equals(n);
	}

	public void deleteIfExists() throws IOException {
		try {
			byte[] key = keyBytes();
			store.db().deleteRange(key, key);
		} catch (RocksDBException e) {
			throw new IOException(e);
		}
	}

	public int countChildren() throws IOException {
		RocksIterator it = store.db().newIterator();
		String prefix = name + ".";
		it.seek(prefix.getBytes());
		int count = 0;
		while(it.isValid()) {
			String key = new String(it.key());
			if(!key.startsWith(prefix))
				break;
			count++;
			it.next();
		}
		return count;
	}

	public void sync() throws IOException {
		// NO NEED
	}

	public void syncChildren() throws IOException {
		// NO NEED
	}

	public void copyTo(AcornKey to) throws IOException {
		try {
			RocksIO io = (RocksIO)getIO();
			byte[] bytes = io.readFully();
			IO target = to.getIO();
			target.saveBytes(bytes, bytes.length, true);
		} catch (RocksDBException e) {
			throw new IOException(e);
		}
	}

	public void deleteAll() throws IOException {
		try {
			RocksIterator it = store.db().newIterator();
			String prefix = name + ".";
			it.seek(prefix.getBytes());
			while(it.isValid()) {
				String key = new String(it.key());
				if(!key.startsWith(prefix))
					break;
				byte[] keyBytes = key.getBytes();
				store.db().deleteRange(keyBytes, keyBytes);
				it.next();
			}
		} catch (RocksDBException e ) {
			throw new IOException(e);
		}
	}

	@Override
	public int hashCode() {
		return Objects.hash(name, parent, store);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		AcornKeyRocks other = (AcornKeyRocks) obj;
		return Objects.equals(name, other.name) && Objects.equals(parent, other.parent)
				&& Objects.equals(store, other.store);
	}

}
