/*******************************************************************************
 * Copyright (c) 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.acorn;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Objects;

import org.simantics.db.IO;

/**
 * @author Antti Villberg
 */
public class AcornKeyFile implements AcornKey {

	private FileStore store;
	private AcornKeyFile parent;
	private String name;
	
	public AcornKeyFile(FileStore store, AcornKeyFile parent, String name) {
		this.store = store;
		this.parent = parent;
		this.name = name;
	}
	
	public IO getIO() {
		return new FileIO(path());
	}
	
	public Path path() {
		if(parent == null) {
			return store.root().resolve(name);
		} else {
			Path parentPath = parent.path();
			return parentPath.resolve(name);
		}
	}
	
	public boolean exists() {
		return Files.exists(path());
	}
	
	public byte[] bytes() throws IOException {
        return Files.readAllBytes(path());
	}
	
	public String getName() {
		return name;
	}
	
	public String debugLocation() {
		return path().toAbsolutePath().toString();
	}
	
	public AcornKey child(String name) {
		return new AcornKeyFile(store, this, name);
	}
	
	public void ensureExists() throws IOException {
		Files.createDirectories(path());
	}
	
	public boolean sameName(String n) {
		return name.equals(n);
	}
	
	public void deleteIfExists() throws IOException {
		Files.deleteIfExists(path());
	}
	
	public int countChildren() throws IOException {
		Path directory = path();
		try (DirectoryStream<Path> ds = Files.newDirectoryStream(directory)) {
			int count = 0;
			for (@SuppressWarnings("unused") Path p : ds)
				++count;
			return count;
		}
	}

	public void sync() throws IOException {
        FileIO.syncPath(path());
	}
	
	public void syncChildren() throws IOException {
		Files.walk(path(), 1).filter(Files::isRegularFile).forEach(FileIO::uncheckedSyncPath);
	}

	public OutputStream newOutputStream() throws IOException {
		return Files.newOutputStream(path());
	}
	
	public void copyTo(AcornKey to) throws IOException {
		Files.copy(path(), ((AcornKeyFile)to).path(), StandardCopyOption.COPY_ATTRIBUTES);
	}

	public void deleteAll() throws IOException {
		Files.walkFileTree(path(), new SimpleFileVisitor<Path>() {
	        @Override
	        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
	            Files.delete(file);
	            return FileVisitResult.CONTINUE;
	        }
	        @Override
	        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
	            Files.delete(dir);
	            return FileVisitResult.CONTINUE;
	        }
	    });
	}

	@Override
	public int hashCode() {
		return Objects.hash(name, parent, store);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		AcornKeyFile other = (AcornKeyFile) obj;
		return Objects.equals(name, other.name) && Objects.equals(parent, other.parent)
				&& Objects.equals(store, other.store);
	}

}
