package org.ththry.fmi.profile.product;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.UUID;
import java.util.logging.Logger;

import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.simantics.fmi.experiment.FMISequenceRunner;
import org.simantics.fmi.experiment.FMUSequenceLoader;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.environment.specification.EnvironmentSpecification;
import org.simantics.scl.compiler.errors.CompilationErrorFormatter;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.module.ImportDeclaration;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.source.FileModuleSource;
import org.simantics.scl.compiler.source.repository.MapModuleSourceRepository;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.util.MultiFunction;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.function.Function;

public class Application implements IApplication {

	private static final String ACTION_KEY = "action";
	private static final String PARAMS_KEY = "params";

	private static Logger logger = Logger.getLogger("org.ththry.fmi.profile.product.Application");
	
    public static EnvironmentSpecification BUILTIN_IMPORTS = EnvironmentSpecification.of(
        "Builtin", "",
        "StandardLibrary", ""
    );
    
    @Override
    public Object start(IApplicationContext context) throws Exception {
    	System.out.println("Starting FMI application...");
        int returnValue = -1;
        try {
        	
            String[] args = (String[])context.getArguments().get("application.args");
            
            if(args.length < 2) {
                logger.severe("At least two parameters, an fmu input file name and script file name, should be given.");
                return IApplication.EXIT_OK;
            }
            
            logger.info("Finding args...");
            
            String fmuFile = args[0];
            
			if(!Files.exists(Paths.get(fmuFile))) {
				throw new FileNotFoundException("Couldn't find model file <" + fmuFile + ">");
			}
            
            logger.info("FMU File: " + fmuFile);
            
            FMUSequenceLoader.loadFMU(fmuFile);
            
            String scriptFile = args[1];
            
			if(!Files.exists(Paths.get(scriptFile))) {
				throw new FileNotFoundException("Couldn't find script file <" + scriptFile + ">");
			}
            
            logger.info("Script File: " + scriptFile);
            
            File file = new File(fmuFile);
            logger.info("FMU file initialized...");
            
            logger.info("Finding action...");
            
            HashMap<String, Object> actionAndParams = actionFromFile(scriptFile, Arrays.copyOfRange(args, 2, args.length));
            
            Object action_obj = actionAndParams.get(ACTION_KEY);
            Object params_obj = actionAndParams.get(PARAMS_KEY);
            
            if(action_obj == null) {
            	System.out.println("Action was null, cannot proceed. Terminating...");
            	return -1;
            }
            
            Function action = (Function)action_obj;
            Object[] params = (Object[])params_obj;
            
            if(params.length > 0) {
            	logger.info("Running action with params...");
            	action.applyArray(params);
            	logger.info("...returned from action run with params.");
            } else {
            	logger.info("Calling FMISequenceRunner.runAction with file and script...");
            	String uuid = UUID.randomUUID().toString();
            	FMISequenceRunner.runActionFromFile(file, uuid, action, true);
            	logger.info("...returned from FMISequenceRunner.runAction.");
            }
            
            returnValue = 0;
        	
            logger.info("Returned from runSCLFromFile");
            return IApplication.EXIT_OK;
        } catch(Throwable t) {
            t.printStackTrace();
            throw (Exception)t;
        } finally {
            System.exit(returnValue);
        }
    }
    
    
    private static HashMap<String, Object> actionFromFile(String filename, String[] parameters) throws IOException {
    	logger.info("Finding the action from file...");
        File file = new File(filename);
        
        if(!file.exists()) {
        	logger.severe("Didn't find " + filename + ".");
            return null;
        }
        
        FileModuleSource source = new FileModuleSource(
                "Main", Application.class.getClassLoader(), file,
                BUILTIN_IMPORTS.imports.toArray(
                        new ImportDeclaration[BUILTIN_IMPORTS.imports.size()]));
        ModuleRepository repository = new ModuleRepository(
                SCLOsgi.MODULE_REPOSITORY,
                new MapModuleSourceRepository(source));
        
        logger.info("Compiling script...");
        Failable<Module> module = repository.getModule("Main");
        if(!module.didSucceed()) {
        	if(module instanceof Failure)
        		System.out.println(CompilationErrorFormatter.toString(source.getSourceText(null), ((Failure)module).errors));
            return null;
        }
        
        SCLValue valueRef = module.getResult().getValue("main");
        if(valueRef == null) {
            System.err.println("Function main is not defined.");
            return null;
        }
        
        try {
            Object value = repository.getValue("Main", "main");
            logger.info("... found the action from the script file.");
            
            HashMap<String, Object> result = new HashMap<>();
            result.put(ACTION_KEY, value);
            
            if(parameters.length > 0) {
            	System.out.println("parameters was > 0");
                Type t = valueRef.getType();
                MultiFunction ft = Types.matchFunction(t, parameters.length);
                Object[] ps = new Object[parameters.length];
                for(int i=0;i<parameters.length;++i) {
                    Type pt = ft.parameterTypes[i];
                    if(Types.equals(pt, Types.DOUBLE))
                        ps[i] = Double.valueOf(parameters[i]);
                    else if(Types.equals(pt, Types.STRING))
                        ps[i] = parameters[i];
                    else if(Types.equals(pt, Types.INTEGER))
                        ps[i] = Integer.valueOf(parameters[i]);
                    else if(Types.equals(pt, Types.FLOAT))
                        ps[i] = Float.valueOf(parameters[i]);
                    else if(Types.equals(pt, Types.BOOLEAN))
                        ps[i] = Boolean.valueOf(parameters[i]);
                    else
                        throw new RuntimeException("Cannot convert command line parameter to type " + pt + ".");
                    logger.info("ps[" + i + "] = " + ps[i] + " :: " + pt + " (converted from \"" + parameters[i] + "\")");
                }
                result.put(PARAMS_KEY, ps);
            } else {
                result.put(PARAMS_KEY, new Object[0]);
            }
            return result;
	    } catch (Exception e) {
            logger.severe(e.getMessage());
            return null;
	    }
    }
    
    @Override
    public void stop() {
    }

}
