/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.impl;

import com.koloboke.function.ShortLongPredicate;
import com.koloboke.function.ShortLongConsumer;
import com.koloboke.collect.map.ShortLongMap;

import java.util.Map;


public final class CommonShortLongMapOps {

    public static boolean containsAllEntries(final InternalShortLongMapOps map,
            Map<?, ?> another) {
        if (map == another)
            throw new IllegalArgumentException();
        if (another instanceof ShortLongMap) {
            ShortLongMap m2 = (ShortLongMap) another;
                if (map.size() < m2.size())
                    return false;
                if (m2 instanceof InternalShortLongMapOps) {
                    //noinspection unchecked
                    return ((InternalShortLongMapOps) m2).allEntriesContainingIn(map);
                }
            return m2.forEachWhile(new
                   ShortLongPredicate() {
                @Override
                public boolean test(short a, long b) {
                    return map.containsEntry(a, b);
                }
            });
        }
        for (Map.Entry<?, ?> e : another.entrySet()) {
            if (!map.containsEntry((Short) e.getKey(),
                    (Long) e.getValue()))
                return false;
        }
        return true;
    }

    public static  void putAll(final InternalShortLongMapOps map,
            Map<? extends Short, ? extends Long> another) {
        if (map == another)
            throw new IllegalArgumentException();
        long maxPossibleSize = map.sizeAsLong() + Containers.sizeAsLong(another);
        map.ensureCapacity(maxPossibleSize);
        if (another instanceof ShortLongMap) {
            if (another instanceof InternalShortLongMapOps) {
                ((InternalShortLongMapOps) another).reversePutAllTo(map);
            } else {
                ((ShortLongMap) another).forEach(new ShortLongConsumer() {
                    @Override
                    public void accept(short key, long value) {
                        map.justPut(key, value);
                    }
                });
            }
        } else {
            for (Map.Entry<? extends Short, ? extends Long> e : another.entrySet()) {
                map.justPut(e.getKey(), e.getValue());
            }
        }
    }

    private CommonShortLongMapOps() {}
}
