/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.compile;

import com.koloboke.collect.hash.ObjHashFactory;

import java.lang.annotation.*;
import java.util.HashMap;
import java.util.HashSet;


/**
 * Specifies whether a Koloboke Compile-generated implementation of the annotated {@code Set}- or
 * {@code Map}-like class or interface with a reference <a
 * href="{@docRoot}/com/koloboke/compile/KolobokeMap.html#model">key type</a> should allow insertion
 * and queries of the {@code null} key. If a {@link KolobokeSet @KolobokeSet}- or {@link
 * KolobokeMap @KolobokeMap}-annotated type is not annotated with {@code @NullKeyAllowed} or
 * annotated with {@code @NullKeyAllowed(false)}, Koloboke Compile generates an implementation that
 * throws {@link NullPointerException} on any attempt to insert or query the presence of the {@code
 * null} key in the instance of this implementation. If the implemented type is annotated with
 * {@code @NullKeyAllowed} or {@code NullKeyAllowed(true)}, Koloboke Compile generates
 * an implementation that fully allows the {@code null} key (just like {@link HashSet} or {@link
 * HashMap}).
 *
 * <p>The Koloboke Collections's API counterpart of {@code @NullKeyAllowed} is {@link
 * ObjHashFactory#withNullKeyAllowed(boolean)} configuration. An important difference is that
 * {@code @NullKeyAllowed} <i>does</i> guarantee that the implementation, generated by Koloboke
 * Compile, throws {@code NullPointerException} to enforce the {@code null} key disallowance in
 * runtime, while {@link ObjHashFactory} <i>doesn't</i> guarantee that.
 *
 * <p>If the annotated type has a numeric primitive key type, Koloboke Compile emits a compilation
 * error and doesn't generate an implementation for the annotated type.
 *
 * @see <a href="{@docRoot}/com/koloboke/compile/KolobokeMap.html#key-nullability">Key nullability
 * section in the <code>@KolobokeMap</code> specification</a>
 * @see <a href="{@docRoot}/com/koloboke/compile/KolobokeSet.html#set-key-nullability">Key
 * nullability section in the <code>@KolobokeSet</code> specification</a>
 */
@Retention(RetentionPolicy.SOURCE)
@Target(ElementType.TYPE)
public @interface NullKeyAllowed {

    /**
     * Tells whether a Koloboke Compile-generated implementation of the annotated {@code Set}- or
     * {@code Map}-like class or interface should allow insertion and queries of the {@code null}
     * key.
     *
     * <p>Default value is {@code true}, that allows natural omission of the value specification:
     * {@code @NullKeyAllowed} is equivalent to {@code @NullKeyAllowed(true)} or
     * {@code @NullKeyAllowed(value=true)}, but {@code @NullKeyAllowed} is not applied to a class
     * or interface altogether, Koloboke Compile generates an implementation that <i>disallow</i>
     * the {@code null} key.
     *
     * @return {@code true} if the Koloboke Compile-generated implementation of the annotated type
     * should allow insertion and queries of the {@code null} key, {@code false} otherwise
     */
    boolean value() default true;
}
