/*
 * Decompiled with CFR 0.152.
 */
package org.simantics.maps.elevation.server;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import java.awt.geom.Rectangle2D;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.geotools.geometry.DirectPosition2D;
import org.geotools.geometry.Envelope2D;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.CRS;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.index.strtree.STRtree;
import org.opengis.geometry.BoundingBox;
import org.opengis.geometry.DirectPosition;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.simantics.maps.elevation.server.TiffInterface;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TiffTileInterface
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TiffTileInterface.class);
    private Path tilesFolder;
    private Map<Path, Envelope> envelopes = new ConcurrentHashMap<Path, Envelope>();
    private LoadingCache<Path, TiffInterface> interfaceCache;
    private int openInterfacesSize;
    private STRtree index;
    private static CoordinateReferenceSystem c4326;

    static {
        try {
            c4326 = CRS.decode((String)"EPSG:4326");
        }
        catch (Exception e) {
            LOGGER.error("Could not initialize EPSG:4326", (Throwable)e);
        }
    }

    public TiffTileInterface(Path tilesFolder) {
        this(tilesFolder, 5);
    }

    public TiffTileInterface(Path tilesFolder, int openInterfacesSize) {
        if (!Files.isDirectory(tilesFolder, new LinkOption[0])) {
            throw new IllegalArgumentException("tilesFolder has to be a folder: " + tilesFolder.toAbsolutePath());
        }
        this.tilesFolder = tilesFolder;
        this.index = new STRtree();
        this.openInterfacesSize = openInterfacesSize;
        this.interfaceCache = Caffeine.newBuilder().maximumSize((long)this.openInterfacesSize).removalListener((key, gdalInterface, cause) -> ((TiffInterface)gdalInterface).close()).build(key -> new TiffInterface((Path)key));
        try {
            this.initializeIndex();
        }
        catch (IOException e) {
            LOGGER.error("Could not initialize index for folder {}", (Object)tilesFolder, (Object)e);
        }
    }

    private TiffInterface openTifInterface(Path tifFile) {
        return (TiffInterface)this.interfaceCache.get((Object)tifFile);
    }

    private Stream<Path> allTiffFiles() throws IOException {
        return Files.walk(this.tilesFolder, new FileVisitOption[0]).filter(path -> Files.isRegularFile(path, new LinkOption[0])).filter(tif -> tif.getFileName().toString().endsWith(".tif"));
    }

    public void initializeIndex() throws IOException {
        LOGGER.info("Initializing index..");
        AtomicInteger counter = new AtomicInteger();
        ((Stream)this.allTiffFiles().parallel()).forEach(tifFile -> {
            TiffInterface tifInterface = this.openTifInterface((Path)tifFile);
            Envelope2D coords = tifInterface.getCornerCoords();
            try {
                try {
                    ReferencedEnvelope refEnv = new ReferencedEnvelope((BoundingBox)coords);
                    ReferencedEnvelope targetEnv = refEnv.transform(c4326, false, 30);
                    STRtree sTRtree = this.index;
                    synchronized (sTRtree) {
                        this.index.insert((Envelope)targetEnv, tifFile);
                    }
                    this.envelopes.put((Path)tifFile, (Envelope)targetEnv);
                }
                catch (Exception e) {
                    LOGGER.error("Could not initialize index for file {}", tifFile, (Object)e);
                    tifInterface.close();
                    int current = counter.getAndIncrement();
                    if (current % 100 == 0) {
                        LOGGER.info("    {}", (Object)current);
                    }
                }
            }
            finally {
                tifInterface.close();
                int current = counter.getAndIncrement();
                if (current % 100 == 0) {
                    LOGGER.info("    {}", (Object)current);
                }
            }
        });
    }

    public Collection<Rectangle2D> getBoundingBoxes() {
        Collection rects = this.envelopes.values().stream().map(env -> {
            double x = env.getMinX();
            double y = env.getMinY();
            return new Rectangle2D.Double(x, y, env.getMaxX() - x, env.getMaxY() - y);
        }).collect(Collectors.toList());
        return rects;
    }

    public Number lookup(double x, double y) {
        LOGGER.trace("Looking up x={} y={}", (Object)x, (Object)y);
        DirectPosition2D p = new DirectPosition2D(c4326, x, y);
        List tifFiles = this.index.query(new Envelope(new Coordinate(x, y)));
        for (Path tifFile : tifFiles) {
            TiffInterface tifInterface = this.openTifInterface(tifFile);
            if (!tifInterface.contains((DirectPosition)p)) continue;
            try {
                Number number = tifInterface.lookup((DirectPosition)p);
                return number;
            }
            finally {
                tifInterface.close();
            }
        }
        return 0.0;
    }

    @Override
    public void close() throws IOException {
        this.interfaceCache.invalidateAll();
        this.interfaceCache.cleanUp();
        this.envelopes.clear();
        this.envelopes = null;
        this.index = null;
        this.interfaceCache = null;
    }
}

