package org.simantics.scl.compiler.constants;

import org.cojen.classfile.TypeDesc;
import org.objectweb.asm.Label;
import org.simantics.scl.compiler.constants.componentaccess.ComponentAccess;
import org.simantics.scl.compiler.constants.componentaccess.FieldComponentAccess;
import org.simantics.scl.compiler.internal.codegen.continuations.Cont;
import org.simantics.scl.compiler.internal.codegen.references.BoundVar;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.references.Val;
import org.simantics.scl.compiler.internal.codegen.types.JavaTypeTranslator;
import org.simantics.scl.compiler.internal.codegen.utils.ClassBuilder;
import org.simantics.scl.compiler.internal.codegen.utils.CodeBuilderUtils;
import org.simantics.scl.compiler.internal.codegen.utils.Constants;
import org.simantics.scl.compiler.internal.codegen.utils.MethodBuilder;
import org.simantics.scl.compiler.types.TVar;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;

public class SCLConstructor extends FunctionValue {
    
    private static int MAX_FIELD_COUNT = Constants.MAX_TUPLE_LENGTH;
    public static ComponentAccess[][] DEFAULT_FIELD_NAMES = new ComponentAccess[MAX_FIELD_COUNT+1][];
    
    static {
        for(int i=0;i<=MAX_FIELD_COUNT;++i) {
        	ComponentAccess[] fieldNames = new ComponentAccess[i];
            for(int j=0;j<i;++j)
                fieldNames[j] = new FieldComponentAccess("c" + j);
            DEFAULT_FIELD_NAMES[i] = fieldNames;
        }
    }
    
    private final String name; // For debugging
    private final String className;
    private final ComponentAccess[] componentAccesses;
    boolean onlyConstructor;
    private final int constructorTag;
       
    public SCLConstructor(String name, String className, TVar[] typeParameters, int constructorTag,
            Type returnType, ComponentAccess[] componentAccesses, Type... parameterTypes) {
        super(typeParameters, Types.NO_EFFECTS, returnType, parameterTypes);
        ClassBuilder.checkClassName(className);
        this.name = name;
        this.className = className;
        this.componentAccesses = componentAccesses;
        this.constructorTag = constructorTag;
    }    
    
    public SCLConstructor(String name, String className, TVar[] typeParameters, int constructorTag,
            Type returnType,
            Type ... parameterTypes) {     
        this(name, className, typeParameters, constructorTag, returnType,
                DEFAULT_FIELD_NAMES[parameterTypes.length], parameterTypes);
    }
    
    public void setOnlyConstructor(boolean onlyConstructor) {
        this.onlyConstructor = onlyConstructor;
    }
        
    @Override
    public Type applyExact(MethodBuilder mb, Val[] parameters) {
        if(className == null) {
            mb.push(parameters[0], parameterTypes[0]);
            return getReturnType();
        }
        else {
            TypeDesc typeDesc = TypeDesc.forClass(className);
            CodeBuilderUtils.constructRecord(typeDesc, mb, 
                    parameterTypes, parameters);
            return getReturnType();
        }
    }
    
    @Override
    public void deconstruct(MethodBuilder mb, IVal parameter, Cont success,
            Label failure) {
        JavaTypeTranslator javaTypeTranslator = mb.getJavaTypeTranslator();

        if(onlyConstructor) {
            if(className == null) {
                TypeDesc parameterDesc = javaTypeTranslator.getTypeDesc(parameter);
                Type expectedType = success.getParameterType(0);
                TypeDesc expectedDesc = javaTypeTranslator.toTypeDesc(expectedType);
                if(parameterDesc.equals(expectedDesc))
                    mb.jump(success, parameter);
                else {
                    parameter.push(mb);
                    mb.unbox(expectedType);
                    BoundVar boundVar = new BoundVar(expectedType);
                    mb.store(boundVar);
                    mb.jump(success, boundVar);
                }
                return;
            }
            failure = null;
        }
        
        Label failureLabel = mb.createLabel();
        
        TypeDesc constructorType = TypeDesc.forClass(className);
        
        // Test if deconstructing is possible
        mb.push(parameter, returnType);
        if(failure != null) {
            mb.dup();
            mb.instanceOf(constructorType);
            mb.ifZeroComparisonBranch(failureLabel, "==");
        }
        
        // Success
        if(!onlyConstructor)
            mb.checkCast(constructorType);
        Val[] parameters = new Val[parameterTypes.length];
        for(int i=0;i<parameterTypes.length;++i) {
            TypeDesc typeDesc = javaTypeTranslator.toTypeDesc(parameterTypes[i]);            
            Type contType = success.getParameterType(i);
            
            if(!typeDesc.equals(TypeDesc.VOID)) {
                mb.dup();
                componentAccesses[i].load(mb, MethodBuilder.getClassName(constructorType), typeDesc);
                if(typeDesc == TypeDesc.OBJECT)
                    mb.unbox(contType);      
            }
            
            BoundVar boundVar = new BoundVar(contType);
            mb.store(boundVar);            
            parameters[i] = boundVar;
        }
        mb.pop();
        mb.jump(success, parameters);
        
        // Failure
        if(failure != null) {
            mb.setLocation(failureLabel);
            mb.pop();
            mb.branch(failure);
        }
    }
    public int constructorTag() {
        return constructorTag;
    }
    
    @Override
    public String toString() {
        return name;
    }
    
}
