/*******************************************************************************
 * Copyright (c) 2007, 2019 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #399
 *******************************************************************************/
package org.simantics.modeling.ui.property;

import java.util.Arrays;
import java.util.Collections;
import java.util.Set;

import org.eclipse.core.expressions.PropertyTester;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.ui.IEditorPart;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.diagramEditor.DiagramViewer;
import org.simantics.ui.SimanticsUI;

/**
 * @author Tuukka Lehtonen
 */
public class TypicalPropertyTester extends PropertyTester {


    /**
     * Tests if the received object is considered deletable.
     */
    private static final String IS_TYPICAL_MASTER_EDITOR = "isMasterEditor";

    /**
     * Tests if the received object is considered modifiable.
     */
    private static final String IS_TYPICAL_INSTANCE_EDITOR = "isInstanceEditor";

    @Override
    public boolean test(final Object receiver, final String property, final Object[] args, final Object expectedValue) {
        //System.out.println("TEST: " + receiver + ", " + property + ", " + Arrays.toString(args) + ", " + expectedValue);

        if (!(receiver instanceof IEditorPart))
            return false;

        if (IS_TYPICAL_MASTER_EDITOR.equals(property)) {
            return isTypicalMasterEditor((IEditorPart) receiver);
        } else if (IS_TYPICAL_INSTANCE_EDITOR.equals(property)) {
            return isTypicalInstanceEditor((IEditorPart) receiver);
        }

        return false;
    }

    private static IDiagram getDiagram(IAdaptable editor) {
        DiagramViewer viewer = editor.getAdapter(DiagramViewer.class);
        return viewer != null ? viewer.getAdapter(IDiagram.class) : null;
    }

    private static Set<String> getDiagramMappedCompositeTypes(IDiagram diagram) {
        Set<String> result = diagram != null ? diagram.getHint(DiagramModelHints.KEY_MAPPED_COMPOSITE_RESOURCE_TYPE_URIS) : null;
        return result != null ? result :  Collections.emptySet();
    }

    private static Set<String> getDiagramMappedCompositeTypes(IAdaptable editor) {
        return getDiagramMappedCompositeTypes( getDiagram(editor) );
    }

    public static boolean isTypicalMasterEditor(IAdaptable editor) {
        Set<String> types = getDiagramMappedCompositeTypes(editor);
        return types.contains(ModelingResources.URIs.MasterTypicalCompositeType);
    }

    private static boolean hasDiagramSource(IDiagram diagram) {
        return diagram.getHint(DiagramModelHints.KEY_HAS_DIAGRAM_SOURCE) != null;
    }

    public static boolean isTypicalInstanceEditor(IAdaptable editor) {
        IDiagram diagram = getDiagram(editor);
        if (diagram == null)
            return false;
        Set<String> types = getDiagramMappedCompositeTypes(diagram);
        return !types.contains(ModelingResources.URIs.MasterTypicalCompositeType)
                && types.contains(ModelingResources.URIs.TypicalComposite)
                && hasDiagramSource(diagram);
    }

    private static boolean timeoutingRead(RequestProcessor processor, Read<Boolean> read) throws DatabaseException, InterruptedException {
        return RequestUtil.trySyncRequest(
                Simantics.getSession(),
                SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                false,
                read);
    }

    public static boolean isTypicalMasterEditor(RequestProcessor processor, Resource editorInputResource) throws DatabaseException, InterruptedException {
        return timeoutingRead(processor, new UniqueRead<Boolean>() {
            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                return isTypicalMasterEditor(graph, editorInputResource);
            }
        });
    }

    public static boolean isTypicalInstanceEditor(RequestProcessor processor, Resource editorInputResource) throws DatabaseException, InterruptedException {
        return timeoutingRead(processor, new UniqueRead<Boolean>() {
            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                return isTypicalInstanceEditor(graph, editorInputResource);
            }
        });
    }

    public static boolean isTypicalMasterEditor(ReadGraph graph, Resource editorInputResource) throws DatabaseException {
        ModelingResources MOD = ModelingResources.getInstance(graph);
        Resource composite = graph.getPossibleObject(editorInputResource, MOD.DiagramToComposite);
        return composite != null
                && graph.isInstanceOf(composite, MOD.MasterTypicalCompositeType);
    }

    public static boolean isTypicalInstanceEditor(ReadGraph graph, Resource editorInputResource) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);
        return graph.isInstanceOf(editorInputResource, DIA.Diagram)
                && graph.hasStatement(editorInputResource, MOD.HasDiagramSource);
    }

}
