/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.preferences;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.preference.FieldEditor;
import org.eclipse.jface.preference.FieldEditorPreferencePage;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.diagram.flag.DiagramFlagPreferences;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ui.Activator;
import org.simantics.utils.datastructures.map.Tuple;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramFlagPreferencePage extends FieldEditorPreferencePage implements IWorkbenchPreferencePage {

    private SchemeSelector       scheme;
    private List<LabelingScheme> schemes;

    public DiagramFlagPreferencePage() {
        super(FLAT);
    }

    @Override
    public void init(IWorkbench workbench) {
    }

    @Override
    protected IPreferenceStore doGetPreferenceStore() {
        return new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
    }

    static class LabelingScheme extends Tuple {
        public LabelingScheme(String label, String description, String uri, Resource scheme) {
            super(label, description, uri, scheme);
        }
        public String getLabel() {
            return (String) getField(0);
        }
        public String getDescription() {
            return (String) getField(1);
        }
        public String getURI() {
            return (String) getField(2);
        }
        public Resource getResource() {
            return (Resource) getField(3);
        }
    }

    static class SchemeSelector extends FieldEditor {

        /**
         * The <code>Combo</code> widget.
         */
        private Combo fCombo;

        /**
         * The value (not the name) of the currently selected item in the Combo widget.
         */
        private LabelingScheme fValue;

        private LabelingScheme fOriginalValue;

        private LabelingScheme fDefaultValue;

        private List<LabelingScheme> fValues;

        /**
         * Create the combo box field editor.
         * 
         * @param label the name of the preference this field editor works on
         * @param labelText the label text of the field editor
         * @param entryNamesAndValues the names (labels) and underlying values to populate the combo widget.  These should be
         * arranged as: { {name1, value1}, {name2, value2}, ...}
         * @param parent the parent composite
         */
        public SchemeSelector(String labelText, List<LabelingScheme> values, LabelingScheme originalValue, LabelingScheme defaultValue, Composite parent) {
            init("labelingScheme", labelText);
            fValues = values;
            fOriginalValue = originalValue;
            fDefaultValue = defaultValue;
            createControl(parent);
        }

        protected void adjustForNumColumns(int numColumns) {
            if (numColumns > 1) {
                Control control = getLabelControl();
                int left = numColumns;
                if (control != null) {
                    ((GridData)control.getLayoutData()).horizontalSpan = 1;
                    left = left - 1;
                }
                ((GridData)fCombo.getLayoutData()).horizontalSpan = left;
            } else {
                Control control = getLabelControl();
                if (control != null) {
                    ((GridData)control.getLayoutData()).horizontalSpan = 1;
                }
                ((GridData)fCombo.getLayoutData()).horizontalSpan = 1;
            }
        }

        protected void doFillIntoGrid(Composite parent, int numColumns) {
            int comboC = 1;
            if (numColumns > 1) {
                comboC = numColumns - 1;
            }
            Control control = getLabelControl(parent);
            GridData gd = new GridData();
            gd.horizontalSpan = 1;
            control.setLayoutData(gd);
            control = getComboBoxControl(parent);
            gd = new GridData();
            gd.horizontalSpan = comboC;
            gd.horizontalAlignment = GridData.FILL;
            control.setLayoutData(gd);
            control.setFont(parent.getFont());
        }

        public int getNumberOfControls() {
            return 2;
        }

        private Combo getComboBoxControl(Composite parent) {
            if (fCombo == null) {
                fCombo = new Combo(parent, SWT.READ_ONLY);
                fCombo.setFont(parent.getFont());
                for (int i = 0; i < fValues.size(); i++) {
                    fCombo.add(fValues.get(i).getLabel(), i);
                }

                fCombo.addSelectionListener(new SelectionAdapter() {
                    public void widgetSelected(SelectionEvent evt) {
                        LabelingScheme oldValue = fValue;
                        String name = fCombo.getText();
                        fValue = getValueForName(name);
                        setPresentsDefaultValue(false);
                        fireValueChanged(VALUE, oldValue, fValue);
                    }
                });
            }
            return fCombo;
        }

        private LabelingScheme getValueForName(String name) {
            for (int i = 0; i < fValues.size(); i++) {
                LabelingScheme scheme = fValues.get(i);
                if (scheme.getLabel().equals(name)) {
                    return scheme;
                }
            }
            return fValues.get(0);
        }

        private void updateComboForValue(LabelingScheme value) {
            fValue = value;
            for (int i = 0; i < fValues.size(); i++) {
                if (value.equals(fValues.get(i))) {
                    fCombo.setText(value.getLabel());
                    return;
                }
            }
            if (fValues.size() > 0) {
                fValue = fValues.get(0);
                fCombo.setText(fValue.getLabel());
            }
        }

        public void setEnabled(boolean enabled, Composite parent) {
            super.setEnabled(enabled, parent);
            getComboBoxControl(parent).setEnabled(enabled);
        }

        @Override
        protected void doLoad() {
            updateComboForValue(fOriginalValue);
        }

        protected void doLoadDefault() {
            updateComboForValue(fDefaultValue);
        }

        @Override
        protected void doStore() {
            if (fValue == null) {
                fOriginalValue = fDefaultValue;
                DiagramFlagPreferences.setProjectFlagLabelingScheme(fDefaultValue.getResource());
                return;
            }
            fOriginalValue = fValue;
            DiagramFlagPreferences.setProjectFlagLabelingScheme(fValue.getResource());
        }

    }

    static class FindSchemes implements Read<List<LabelingScheme>> {
        @Override
        public List<LabelingScheme> perform(ReadGraph graph) throws DatabaseException {
            DiagramResource DIA = DiagramResource.getInstance(graph);
            Layer0 L0 = Layer0.getInstance(graph);
            List<LabelingScheme> result = new ArrayList<LabelingScheme>();
            for (Resource scheme : graph.syncRequest(new ObjectsWithType(DIA.FlagLabelingScheme, L0.ConsistsOf, DIA.FlagLabelingScheme))) {
                String label = graph.adapt(scheme, String.class);
                String description = graph.getPossibleRelatedValue(scheme, L0.HasDescription);
                String uri = graph.getURI(scheme);
                result.add(new LabelingScheme(label, description, uri, scheme));
            }
            return result;
        }
    }

    @Override
    protected void createFieldEditors() {
        schemes = Collections.emptyList();
        Resource schemeValue = null;
        try {
            schemes = Simantics.getSession().syncRequest(new FindSchemes());
            schemeValue = DiagramFlagPreferences.getActiveFlagLabelingSchemeResource(Simantics.getSession());
        } catch (DatabaseException e) {
            ExceptionUtils.logAndShowError(e);
        }

        LabelingScheme previousValue = null;
        LabelingScheme defaultValue = null;

        for (int i = 0; i < schemes.size(); ++i) {
            LabelingScheme s = schemes.get(i);
            if (DiagramResource.URIs.FlagLabelingScheme_Alphabetical.equals(s.getURI()))
                defaultValue = s;
        }
        if (defaultValue == null && !schemes.isEmpty())
            defaultValue = schemes.get(0);

        if (schemeValue != null) {
            for (int i = 0; i < schemes.size(); ++i) {
                LabelingScheme s = schemes.get(i);
                if (schemeValue.equals(s.getResource()))
                    previousValue = s;
            }
        }
        if (previousValue == null)
            previousValue = defaultValue;

        scheme = new SchemeSelector("Labeling Scheme", schemes, previousValue, defaultValue, getFieldEditorParent());
        addField(scheme);
    }

}
