/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.genericfileimport.wizard;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Deque;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.Simantics;
import org.simantics.db.management.ISessionContext;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Antti Villberg
 */
public class GenericFileImportWizard extends Wizard implements IImportWizard {

    private static final int MAX_RECENT_IMPORT_PATHS = 10;

    GenericFileImportPlan        importModel;

    private boolean readPreferences(IStructuredSelection selection) {

        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        String recentPathsPref = store.getString(Preferences.RECENT_GENERIC_FILE_IMPORT_LOCATIONS);
        Deque<String> recentImportPaths = Preferences.decodePaths(recentPathsPref);

        ISessionContext ctx = Simantics.getSessionContext();
        if (ctx == null)
            return false;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return false;

        importModel = new GenericFileImportPlan(ctx, recentImportPaths);
        importModel.project = project;

        return true;

    }

    private void writePreferences() throws IOException {

    	IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(Preferences.RECENT_GENERIC_FILE_IMPORT_LOCATIONS, Preferences.encodePaths(importModel.recentLocations));

        if (store.needsSaving())
            store.save();

    }

    public GenericFileImportWizard() {
        setWindowTitle("Import File");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences(selection);
    }

    @Override
    public void addPages() {
        super.addPages();
        if (importModel != null) {
            addPage(new GenericFileImportPage(importModel));
        } else {
            addPage(new NoProjectPage("Import File"));
        }
    }

    @Override
    public boolean performFinish() {
        try {
            importModel.recentLocations.addFirst(importModel.importLocation.getAbsolutePath());
            Preferences.removeDuplicates(importModel.recentLocations);
            if (importModel.recentLocations.size() > MAX_RECENT_IMPORT_PATHS)
                importModel.recentLocations.pollLast();

            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        try {

        	getContainer().run(true, true, monitor -> {
                try {
                    importModel.sessionContext.getSession().markUndoPoint();
                    Simantics.applySCL("Dropins/Core", "importGenericFileToResource2", importModel.importLocation.getAbsolutePath());
                } catch (Exception e) {
                    throw new InvocationTargetException(e);
                } finally {
                    monitor.done();
                }
            });

        } catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            WizardPage cp = (WizardPage) getContainer().getCurrentPage();
            cp.setErrorMessage("Unexpected problem importing model.\nMessage: " + cause.getMessage());
            ErrorLogger.defaultLogError("File " + importModel.importLocation + " import failed unexpectedly. See exception for details.", cause);
            ExceptionUtils.showError("Unexpected problem importing file.\n\n" + cause.getMessage(), cause);
            return false;
        } catch (InterruptedException e) {
            WizardPage cp = (WizardPage) getContainer().getCurrentPage();
            cp.setErrorMessage("Import interrupted.\nMessage: " + e.getMessage());
            ErrorLogger.defaultLogError("File " + importModel.importLocation + " import interrupted.", e);
            ExceptionUtils.showError("File import was interrupted.", e);
            return false;
        }

        return true;
    }

}
