/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history.util;

/**
 * A custom progress monitor interface for long operations (not using
 * Eclipse IProgressMonitor to avoid dependency on Eclipse runtime plug-ins).
 * 
 * @author Tuukka Lehtonen
 */
public interface ProgressMonitor {

    /**
     * Notifies that the main task is beginning.  This must only be called once
     * on a given progress monitor instance.
     * 
     * @param name the name (or description) of the main task
     * @param totalWork the total number of work units into which
     *  the main task is been subdivided. If the value is <code>UNKNOWN</code> 
     *  the implementation is free to indicate progress in a way which 
     *  doesn't require the total number of work units in advance.
     */
    public void beginTask(String name, int totalWork);

    /**
     * Returns whether cancelation of current operation has been requested.
     * Long-running operations should poll to see if cancelation
     * has been requested.
     *
     * @return <code>true</code> if cancellation has been requested,
     *    and <code>false</code> otherwise
     */
    public boolean isCanceled();

    /**
     * Sets the task name to the given value. This method is used to 
     * restore the task label after a nested operation was executed. 
     * Normally there is no need for clients to call this method.
     *
     * @param name the name (or description) of the main task
     * @see #beginTask(java.lang.String, int)
     */
    public void setTaskName(String name);

    /**
     * Notifies that a subtask of the main task is beginning.
     * Subtasks are optional; the main task might not have subtasks.
     *
     * @param name the name (or description) of the subtask
     */
    public void subTask(String name);

    /**
     * Notifies that a given number of work unit of the main task
     * has been completed. Note that this amount represents an
     * installment, as opposed to a cumulative amount of work done
     * to date.
     *
     * @param work a non-negative number of work units just completed
     */
    public void worked(int work);

    /**
     * @author Tuukka Lehtonen
     */
    public static class Stub implements ProgressMonitor {
        @Override
        public void beginTask(String name, int totalWork) {
        }
        @Override
        public boolean isCanceled() {
            return false;
        }
        @Override
        public void setTaskName(String name) {
        }
        @Override
        public void subTask(String name) {
        }
        @Override
        public void worked(int work) {
        }
    }
    
}