/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.simantics.databoard.binding.Binding;
import org.simantics.history.impl.CollectorImpl;
import org.simantics.history.impl.FileHistory;
import org.simantics.history.impl.FlushPolicy;
import org.simantics.history.impl.MemoryHistory;
import org.simantics.history.util.HistoryExportUtil;

/**
 * Facade-class for History Services.
 * 
 * There are two main services HistoryManager and Collector.
 * There are two main implementations file and memory.
 * 
 * @author toni.kalajainen
 */
public class History {

	/**
	 * Open a file based history manager.
	 * (Note, Trend does not need flush policy)
	 * 
	 * @param workarea
	 * @return history manager
	 */
	public static HistoryManager openFileHistory(File workarea)
	{
		return new FileHistory(workarea);
	}
	
	/**
	 * Create a transient memory history
	 * 
	 * @return no history
	 */
	public static HistoryManager createMemoryHistory()
	{
		return new MemoryHistory();
	}
	
	/**
	 * Create collector
	 * 
	 * @param historyManager
	 * @return collector
	 */
	public static Collector createCollector(HistoryManager historyManager)
	{
		CollectorImpl result = new CollectorImpl(historyManager);
		return result;
	}
	
	/**
	 * Create collector
	 * 
	 * @param historyManager
	 * @param flushPolicy
	 * @return collector
	 */
	public static Collector createCollector(HistoryManager historyManager, FlushPolicy flushPolicy)
	{
		CollectorImpl result = new CollectorImpl(historyManager);
		result.setFlushPolicy(flushPolicy);
		return result;
	}

	/**
	 * Export whole history manager to an output stream.
	 * 
	 * @param history
	 * @param timeBinding
	 * @param from
	 * @param end
	 * @param os
	 * @throws IOException
	 * @throws HistoryException
	 */
	public static void exportHistory( HistoryManager history, Binding timeBinding, Double from, Double end, OutputStream os )
	throws IOException, HistoryException
	{
		HistoryExportUtil eu = new HistoryExportUtil();
		eu.exportHistory(history, timeBinding, from, end, os);
	}
	
	/**
	 * Import a history manager from an input stream 
	 * 
	 * @param history
	 * @param is
	 * @throws IOException
	 * @throws HistoryException
	 */
	public static void importHistory( HistoryManager history, InputStream is )
	throws IOException, HistoryException
	{
		HistoryExportUtil eu = new HistoryExportUtil();
		eu.importHistory(history, is);
	}

}
