/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.procedure.guarded;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.common.procedure.adapter.AsyncProcedureAdapter;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Procedure;

/**
 * An asynchronous database callback procedure that is guarded against multiple
 * invocations to {@link Procedure#execute(Object)} and
 * {@link Procedure#exception(Throwable)}. Only the first invocation of is
 * delegated or the Nth invocation to {@link Procedure#execute(Object)} where
 * Nth is <code>expectedResults</code>.
 * 
 * @author Tuukka Lehtonen
 * 
 * @param <T>
 */
public class GuardedAsyncProcedureWrapper<T> extends AsyncProcedureAdapter<T> {

    private final AsyncProcedure<T> procedure;
    private final AtomicBoolean     onceGuard = new AtomicBoolean(false);
    private final AtomicInteger     resultCounter;

    public GuardedAsyncProcedureWrapper(AsyncProcedure<T> procedure, int expectedResults) {
        this.procedure = procedure;
        this.resultCounter = new AtomicInteger(expectedResults);
    }

    @Override
    public void exception(AsyncReadGraph graph, Throwable t) {
        if (onceGuard.compareAndSet(false, true)) {
            procedure.exception(graph, t);
        }
    }

    @Override
    public void execute(AsyncReadGraph graph, T result) {
        int count = resultCounter.decrementAndGet();
        if (count == 0) {
            if (onceGuard.compareAndSet(false, true)) {
                procedure.execute(graph, result);
            }
        } else if (count < 0) {
            System.out.println(this + ": execute invoked more than expected (count=" + count + ", procedure=" + procedure + ")");
        }
    }

}
