/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.DoubleBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.IdentityPair;

/**
 * Binds Double[] type to double[] Class.
 */
public final class DoubleArrayBinding extends ArrayBinding {

	public static DoubleArrayBinding createFrom(ArrayType type) {
		return new DoubleArrayBinding(type, new DoubleBindingDefault((DoubleType) type.componentType));
	}
	
	public DoubleArrayBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
	}
	
	@Override
	public Object create() {
		return new double[0];
	}

	@Override
	public Object create(int length, Iterator<Object> values) {
		double[] array = new double[length];
		for(int i=0;i<length;++i)
			array[i] = (Double)values.next();
		return array;
	}
	
	public Object create(Object[] values) {
		double[] result = new double[values.length];
		for (int i=0; i<values.length; i++)
			result[i] = (Double) values[i];
		return result;
	}
	
	@Override
    public Object createDefault()
    throws BindingException
    {
		NumberType nt = (NumberType) type().componentType;
		if (nt.getRange() == null) {
	    	return new double[ type().minLength() ];
		}
		return super.createDefault();
    }	
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		DoubleBinding scb = (DoubleBinding) sb.getComponentBinding();
		
		double[] d = (double[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("double[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
	}	

	@Override
	public Object readFromTry(Binding srcBinding, Object src, Object dst) throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		DoubleBinding scb = (DoubleBinding) sb.getComponentBinding();
		
		double[] d = (double[]) dst;
		int srcSize = sb.size(src);
		if (d.length != srcSize) d = new double[ srcSize ];
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
		
		return d;
	}	

	/**
	 * Create an array object.
	 * Note! The argument is consumed (is used in the result).
	 * 
	 * @param array
	 * @return an object that contains the array
	 */
	public Object create(double[] array) {
		return array;
	}
	
	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", double[] expected");
		return ((double[])array)[index];
	}

	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		double[] a = (double[])array;
		for (int i=0; i<a.length; i++)
			result[i] = a[i];
	}
	
	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		double[] a = (double[])array;
		a[index] = (Double) value;
	}
	
	@Override
	public int size(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", double[] expected");
		return ((double[])array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof double[];
	}
	
	@Override
	public boolean isImmutable() {
		return false;
	}	
	
	public double[] getArray(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", double[] expected");
		return (double[]) array;
	}	

    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	double[] array = (double[]) value;
    	for (int i=0; i<array.length; i++) {
    		double element = array[i];    		
    		long bits = Double.doubleToLongBits(element);
    		int elementHash = (int)(bits ^ (bits >>> 32));     		
    		result = 31*result + elementHash;
    	}
    	return result;
    }
	
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	double[] a1 = (double[]) o1;
    	double[] a2 = (double[]) o2;
		// Compare Lengths
		int l1 = a1.length;
		int l2 = a2.length;
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		for (int i=0; i<l1; i++) {
			double v1 = a1[i];
			double v2 = a2[i];
			dif = Double.compare(v1, v2);
			if (dif!=0) return dif;
		}
		return 0;
    }    

    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
    	throw new UnsupportedOperationException();
    }
    
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
    	throw new UnsupportedOperationException();
	}
    
	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("double[] is length immutable");
	}	

	@Override
	public boolean isResizable() {
		return false;
	}
	
}
