/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.NodeContext.PrimitiveQueryKey;
import org.simantics.browsing.ui.PrimitiveQueryProcessor;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.Container;

public abstract class LazyResourceQueryContainer<Result> implements Container<Result> {

    final private ResourceQuery<Result> query;

    final private PrimitiveQueryUpdater updater;

    private final Listener<Result> procedure;

    final protected NodeContext        context;

    private Result                      result;

    private boolean                     computed = false;

    /**
     * Computes the graph query result. This will get called asynchronously.
     * 
     * @param graph
     * @return
     */
    protected abstract Result compute(ReadGraph graph) throws DatabaseException;

    /**
     * Needed for retrieving the actual primitive query key that is used to with
     * {@link PrimitiveQueryUpdater#scheduleReplace(NodeContext, PrimitiveQueryKey, Object)}
     * inside the {@link #query} that is initialized in the constructor
     * {@link #LazyResourceQueryContainer(PrimitiveQueryUpdater, NodeContext, Object, Object)}
     * .
     * 
     * <p>
     * This key should originally be received by the actual
     * {@link PrimitiveQueryProcessor} that has been invoked to compute the
     * result that will be stored in this {@link LazyResourceQueryContainer}.
     * 
     * @return
     */
    protected abstract PrimitiveQueryKey<Container<Result>> getKey();

    public LazyResourceQueryContainer(final PrimitiveQueryUpdater updater, final NodeContext context, Result initial) {

        this.updater = updater;
        this.context = context;
        this.result = initial;

        this.query = new ResourceQuery<Result>(getKey(), context) {

            @Override
            public Result perform(ReadGraph graph) throws DatabaseException {
                return compute(graph);
            }

        };

        procedure = new Listener<Result>() {

            @Override
            public void execute(Result result) {
                setResult(result);
                updater.scheduleReplace(context, getKey(), LazyResourceQueryContainer.this);
            }

            @Override
            public boolean isDisposed() {
                return updater.isDisposed();
            }

            public void exception(Throwable t) {
                System.out.print("LazyResourceQueryContainer2.request failed: ");
                t.printStackTrace();
            }

        };

    }

    protected PrimitiveQueryUpdater getUpdater() {
        return updater;
    }

    private void setResult(Result result) {
        this.result = result;
        computed = true;
    }

    @Override
    public Result get() {

        if (!computed) {

            final DataSource<AsyncReadGraph> source = updater.getDataSource(AsyncReadGraph.class);
            assert(source != null);

            source.schedule(graph -> graph.asyncRequest(query, procedure));

        }

        return result;

    }

}
