/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.Collections;
import java.util.Map;

import org.simantics.browsing.ui.BuiltinKeys.LabelerKey;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.labelers.LabelerContent;
import org.simantics.browsing.ui.common.labelers.LabelerStub;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.ui.ErrorLogger;
import org.slf4j.Logger;

public abstract class LazyGraphLabeler extends LabelerStub {

    protected final PrimitiveQueryUpdater       updater;
    private final ResourceQuery<LabelerContent> labelQuery;
    private final Listener<LabelerContent>      procedure;

    public Object getIdentity(LabelerKey key) {
        return key;
    }

    public LazyGraphLabeler(final PrimitiveQueryUpdater updater, final NodeContext context, final LabelerKey key) {
        this.updater = updater;

        labelQuery = new ResourceQuery<LabelerContent>(getIdentity(key), context) {

            @Override
            public LabelerContent perform(ReadGraph graph) throws DatabaseException {
                try {
                    int cat = category(graph);
                    Map<String, String> lbls = labels(graph);
                    // Make sure that null is not returned.
                    if (lbls == null)
                        throw new NullPointerException("LazyGraphLabeler.labels is not allowed to return null, but " + LazyGraphLabeler.this.getClass() + " just did it");
                    return new LabelerContent(cat, lbls);
                } catch (DatabaseException e) {
                    throw e;
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError("LazyGraphLabeler.labelQuery produced unexpected exception.", t);
                    return LabelerContent.NO_CONTENT;
                }
            }

            @Override
            public String toString() {
                return LazyGraphLabeler.this + " with context " + context;
            }

        };

        procedure = new Listener<LabelerContent>() {

            @Override
            public void execute(LabelerContent result) {
                setContent(result);
                updater.scheduleReplace(context, key, LazyGraphLabeler.this);
            }

            @Override
            public boolean isDisposed() {
                return updater.isDisposed();
            }

            @Override
            public void exception(Throwable t) {
                getLogger().error("LazyGraphLabeler2: ", t);
            }

        };
    }

    @Override
    public Map<String, String> getLabels() {

        if (content == LabelerContent.NO_CONTENT) {

            final DataSource<AsyncReadGraph> source = updater.getDataSource(AsyncReadGraph.class);
            assert(source != null);
            source.schedule(graph -> graph.asyncRequest(labelQuery, procedure));

        }

        if(content == null) return Collections.emptyMap();

        return content.labels;

    }

    // OVERRIDE

    public abstract Map<String, String> labels(ReadGraph graph) throws DatabaseException;

    public int category(ReadGraph graph) throws DatabaseException {

        if(content == null) return 0;

        return content.category;

    }

    public abstract Logger getLogger();
}
