package org.simantics.acorn.lru;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;

import org.simantics.acorn.FileCache;
import org.simantics.acorn.exception.AcornAccessVerificationException;
import org.simantics.acorn.exception.IllegalAcornStateException;
import org.simantics.db.service.Bytes;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.list.array.TByteArrayList;

public class ChangeSetInfo extends LRUObject<Long, ChangeSetInfo> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ChangeSetInfo.class);
	private byte[] metadataBytes;
	private ArrayList<String> clusterChangeSetIds;
	
	// Stub
	public ChangeSetInfo(LRU<Long, ChangeSetInfo> LRU, FileCache fileCache, Path readDir, Long revision, int offset, int length) throws AcornAccessVerificationException {
		super(LRU, fileCache, revision, readDir, "clusterStream", offset, length, false, false);
		LRU.map(this);
	}
	
	// New
	public ChangeSetInfo(LRU<Long, ChangeSetInfo> LRU, FileCache fileCache, Long revision, byte[] bytes, ArrayList<String> clusterChangeSetIds) throws AcornAccessVerificationException {
		super(LRU, fileCache, revision, LRU.getDirectory(), "clusterStream", true, true);
		this.metadataBytes = bytes;
		this.metadataBytes = bytes;
		this.clusterChangeSetIds = clusterChangeSetIds;
		LRU.insert(this, accessTime);
	}
	
	public ArrayList<String> getCCSIds() throws AcornAccessVerificationException {
		if(VERIFY) verifyAccess();
		return clusterChangeSetIds;
	}
	
	public byte[] getMetadataBytes() throws AcornAccessVerificationException, IllegalAcornStateException {
		if(VERIFY)
		    verifyAccess();
		
		makeResident();
		return metadataBytes;
	}
	
	private static void writeLE(TByteArrayList bytes, int value) {
		
		bytes.add( (byte) (value & 0xFF));
		bytes.add((byte) ((value >>> 8) & 0xFF));
		bytes.add((byte) ((value >>> 16) & 0xFF));
		bytes.add((byte) ((value >>> 24) & 0xFF));

	}

	@Override
	protected Pair<byte[], Integer> toBytes() {
		
		TByteArrayList result = new TByteArrayList();
		writeLE(result, metadataBytes.length);
		result.add(metadataBytes);
		writeLE(result, clusterChangeSetIds.size());
		for(String id : clusterChangeSetIds) {
			byte[] bb = id.getBytes();
			writeLE(result, bb.length);
			result.add(bb);
		}

		release();
		
		byte[] ret = result.toArray();
		
		return Pair.make(ret, ret.length);
		
	}
	
	@Override
	void release() {
		clusterChangeSetIds = null;
		metadataBytes = null;
	}

	@Override
	public void fromFile(byte[] data) {
		
		clusterChangeSetIds = new ArrayList<String>();
		
		int metadataLength = Bytes.readLE4(data, 0);
		metadataBytes = Arrays.copyOfRange(data, 4, 4+metadataLength);
		int offset = 4+metadataLength;
		int numberOfChangeSets = Bytes.readLE4(data, offset);
		offset += 4;
		for(int i=0;i<numberOfChangeSets;i++) {
			int length = Bytes.readLE4(data, offset);
			offset += 4;
			String id = new String(Arrays.copyOfRange(data, offset, offset+length));
			clusterChangeSetIds.add(id);
			offset += length;
		}
		
	}

	@Override
	String getExtension() {
		return "cs";
	}

	@Override
	public String toString() {
		return  "ChangeSetInfo " + getKey();
	}

	@Override
	protected boolean overwrite() {
		return false;
	}

	@Override
	public Logger getLogger() {
	    return LOGGER;
	}
}