package org.simantics.views;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.util.Bean;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.TernaryRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.Functions;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.procedure.Listener;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.scl.runtime.function.Function2;
import org.simantics.views.ontology.ViewsResources;

public class ViewUtils {

	public static class ExtendedMargins {
		
        public int left, right, top, bottom;
		
    }

    public static final Binding EXTENDED_MARGINS_BINDING = Bindings.getBindingUnchecked(ExtendedMargins.class);

	public static class LayoutBean extends Bean {
		public int marginLeft;
		public int marginRight;
		public int marginTop;
		public int marginBottom;
	}

	public static class GridLayoutBean extends LayoutBean {
		
		public int numColumns;
		public int horizontalSpacing;
		public int verticalSpacing;
		
	}

	public static class RowLayoutBean extends LayoutBean {
		public int type;
		public int spacing;
		public boolean center;
		public boolean fill;
		public boolean justify;
		public boolean pack;
		public boolean wrap;
	}

	public static class LayoutDataBean extends Bean {}

	public static class GridDataBean extends LayoutDataBean {
		
		public int horizontalSpan;
		public boolean grabExcessHorizontalSpace;
		public boolean grabExcessVerticalSpace;
		public int horizontalAlignment;
		public int verticalAlignment;
		public int widthHint;
		public int heightHint;
		
	}

	public static class RowDataBean extends LayoutDataBean {
		public int width;
		public int height;
	}

	public static class ColumnBean extends Bean {
		
		public String key;
		public String label;
		public String alignment;
		public int width;
		public String tooltip;
		public boolean grab;
		public int weight;
		
	}

	private static void readMargins(ExtendedMargins extendedMargins, LayoutBean layout) {
		if (extendedMargins != null) {
			layout.marginLeft = extendedMargins.left;
			layout.marginRight = extendedMargins.right;
			layout.marginTop = extendedMargins.top;
			layout.marginBottom = extendedMargins.bottom;
		} else {
			layout.marginLeft = 3; 
			layout.marginRight = 3;
			layout.marginTop = 3;
			layout.marginBottom = 3;
		}
	}

	public static GridLayoutBean getLayout(RequestProcessor processor, Resource configuration) throws DatabaseException {

		return processor.sync(new ResourceRead<GridLayoutBean>(configuration) {

			@Override
			public GridLayoutBean perform(ReadGraph graph) throws DatabaseException {

				ViewsResources VIEW = ViewsResources.getInstance(graph);

				Integer columns = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_columnCount, Bindings.INTEGER);
				Integer horizontalSpacing = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_horizontalSpacing, Bindings.INTEGER);
				Integer verticalSpacing = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_verticalSpacing, Bindings.INTEGER);
				ExtendedMargins extendedMargins = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_extendedMargins, EXTENDED_MARGINS_BINDING);
				
				GridLayoutBean layout = new GridLayoutBean();//.fillDefaults().numColumns(1).equalWidth(false).margins(0, 0).spacing(0, 0).create();
				layout.numColumns = columns != null ? columns : 1;
				layout.horizontalSpacing = horizontalSpacing != null ? horizontalSpacing : 5;
				layout.verticalSpacing = verticalSpacing != null ? verticalSpacing : 5;
				readMargins(extendedMargins, layout);

				return layout;

			}

		});
		
	}

	public static GridDataBean getGridData(RequestProcessor processor, Resource configuration) throws DatabaseException {
	
		return processor.sync(new ResourceRead<GridDataBean>(configuration) {

			@Override
			public GridDataBean perform(ReadGraph graph) throws DatabaseException {

				ViewsResources VIEW = ViewsResources.getInstance(graph);
				GridDataBean data = new GridDataBean();
				
				data.horizontalSpan = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_horizontalSpan, Bindings.INTEGER);
				data.grabExcessHorizontalSpace = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_horizontalGrab, Bindings.BOOLEAN);
				data.grabExcessVerticalSpace = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_verticalGrab, Bindings.BOOLEAN);
				data.horizontalAlignment = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_horizontalAlignment, Bindings.INTEGER);
				data.verticalAlignment = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_verticalAlignment, Bindings.INTEGER);
				data.widthHint = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_preferredWidth, Bindings.INTEGER);
				data.heightHint = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_preferredHeight, Bindings.INTEGER);

				return data;
				
			}

		});
    
	}

	public static RowLayoutBean getRowLayout(RequestProcessor processor, Resource configuration) throws DatabaseException {
	    return processor.sync(new ResourceRead<RowLayoutBean>(configuration) {
	        @Override
	        public RowLayoutBean perform(ReadGraph graph) throws DatabaseException {
	            ViewsResources VIEW = ViewsResources.getInstance(graph);
	            Integer type = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_type, Bindings.INTEGER);
	            Integer spacing = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_spacing, Bindings.INTEGER);
	            Boolean center = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_center, Bindings.BOOLEAN);
                Boolean fill = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_fill, Bindings.BOOLEAN);
                Boolean justify = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_justify, Bindings.BOOLEAN);
	            Boolean pack = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_pack, Bindings.BOOLEAN);
	            Boolean wrap = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_wrap, Bindings.BOOLEAN);
	            ExtendedMargins extendedMargins = graph.getPossibleRelatedValue(resource, VIEW.RowLayout_extendedMargins, EXTENDED_MARGINS_BINDING);

	            RowLayoutBean layout = new RowLayoutBean();
	            layout.type = type != null ? type : 256;
	            layout.spacing = spacing != null ? spacing : 3;
                layout.center = center != null ? center : false;
                layout.fill = fill != null ? fill : false;
                layout.justify = justify != null ? justify : false;
	            layout.pack = pack != null ? pack : true;
	            layout.wrap = wrap != null ? wrap : true;
	            readMargins(extendedMargins, layout);
	            return layout;
	        }
	    });
	}

	public static RowDataBean getRowData(RequestProcessor processor, Resource configuration) throws DatabaseException {
	    return processor.sync(new ResourceRead<RowDataBean>(configuration) {
	        @Override
	        public RowDataBean perform(ReadGraph graph) throws DatabaseException {
	            ViewsResources VIEW = ViewsResources.getInstance(graph);
	            RowDataBean data = new RowDataBean();
	            Integer width = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_preferredWidth, Bindings.INTEGER);
	            Integer height = graph.getPossibleRelatedValue(resource, VIEW.GridLayout_GridData_preferredHeight, Bindings.INTEGER);
	            data.width = width != null ? width : -1;
	            data.height = height != null ? height : -1;
	            return data;
	        }
	    });
	}

	public static int getStyle(RequestProcessor processor, Resource configuration) throws DatabaseException {
		
		return processor.sync(new ResourceRead<Integer>(configuration) {

			@Override
			public Integer perform(ReadGraph graph) throws DatabaseException {

				ViewsResources VIEW = ViewsResources.getInstance(graph);
				
				int result = 0;
				for(Resource constant : graph.getObjects(resource, VIEW.Control_Style_HasConstant)) {
					int value = graph.getValue(constant, Bindings.INTEGER);
					result |= value;
				}
				
				return result;
				
			}

		});
    
	}

	public static ColumnBean getColumn(RequestProcessor processor, Resource configuration) throws DatabaseException {
		
		return processor.sync(new ResourceRead<ColumnBean>(configuration) {

			@Override
			public ColumnBean perform(ReadGraph graph) throws DatabaseException {

				ViewsResources VIEW = ViewsResources.getInstance(graph);
				String key = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasKey, Bindings.STRING);
				String label = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasLabel, Bindings.STRING);
				String alignment = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasAlignment, Bindings.STRING);
				Integer width = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasWidth, Bindings.INTEGER);
				String tooltip = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasTooltip, Bindings.STRING);
				Boolean grab = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasGrab, Bindings.BOOLEAN);
				Integer weight = graph.getPossibleRelatedValue(resource, VIEW.Explorer_Column_HasWeight, Bindings.INTEGER);
				
				ColumnBean bean = new ColumnBean();
				bean.key = key;
				bean.label = label;
				bean.alignment = alignment;
				bean.width = width != null ? width : -1;
				bean.tooltip = tooltip;
				bean.grab = grab != null ? grab : false;
				bean.weight = weight != null ? weight : 0;
				
				return bean;
				
			}
			
		});
    
	}
	
	public static <T> void listen(Resource configuration, Variable context, String relationURI, final Binding binding, final Function1<T, Boolean> function) throws DatabaseException {

		Simantics.getSession().async(new TernaryRead<Resource, Variable, String, T> (configuration, context, relationURI) {

			@SuppressWarnings("unchecked")
			@Override
			public T perform(ReadGraph graph) throws DatabaseException {
				Object value = graph.getRelatedValue2(parameter, graph.getResource(parameter3), parameter2);
				Object result = binding.createDefaultUnchecked();
				try {
					binding.readFrom(Bindings.getBinding(binding.type()), value, result);
				} catch (BindingException e) {
//					e.printStackTrace();
					throw new DatabaseException(e);
				}
				return (T)result;
			}
			
		}, new Listener<T>() {

			private boolean disposed = false;
			
            @Override
            public void exception(Throwable t) {
//            	t.printStackTrace();
            }

            @Override
            public void execute(T result) {
            	disposed = function.apply(result);
            }

            @Override
            public boolean isDisposed() {
                return disposed;
            }

        });
    
	}
	
	public static <T> void listen(Resource configuration, Variable context, String relationURI, final Function1<T, Boolean> function) throws DatabaseException {

		Simantics.getSession().async(new TernaryRead<Resource, Variable, String, T> (configuration, context, relationURI) {

			@Override
			public T perform(ReadGraph graph) throws DatabaseException {
	        	return graph.getRelatedValue2(parameter, graph.getResource(parameter3), parameter2);
			}
			
		}, new Listener<T>() {

			private boolean disposed = false;
			
            @Override
            public void exception(Throwable t) {
//            	t.printStackTrace();
            }

            @Override
            public void execute(T result) {
            	disposed = function.apply(result);
            }

            @Override
            public boolean isDisposed() {
                return disposed;
            }

        });
    
	}
	
	public static Function2<Object, Object, Object> getActionFunctionDeprecated(final Resource configuration, final Resource runtime, final String relationURI) throws DatabaseException {
		
		return new Function2<Object, Object, Object>() {
			
			@Override
			public Object apply(final Object selection, final Object event) {
				
				Simantics.getSession().async(new WriteRequest() {

					@Override
					public void perform(WriteGraph graph) throws DatabaseException {

			        	Resource relation = graph.getResource(relationURI);
			        	
			        	final Resource function = graph.getSingleObject(configuration, relation);
						
						Functions.exec(graph, function, graph, runtime, selection, event);
						
					}
					
				});
				
				return null;
				
			}
			
		};
    
	}
	
	public static void setWorkbenchSelection(ISelection selection) {
		ISelectionProvider provider = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActivePart().getSite().getSelectionProvider();
        provider.setSelection(selection);
	}
	
}
