/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.export.core.pdf;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.simantics.export.core.internal.Activator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.lowagie.text.Document;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.PdfPageEventHelper;
import com.lowagie.text.pdf.PdfWriter;

/**
 * @author Tuukka Lehtonen
 * @since 1.22.2
 */
public class ServiceBasedPdfExportPageEvent extends PdfPageEventHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServiceBasedPdfExportPageEvent.class);

	Map<String, PdfExportPageEvent> events;
	
	public ServiceBasedPdfExportPageEvent() {
		// Read all page event contributions from the OSGi service
		events = collectEvents();
	}

	private static Map<String, PdfExportPageEvent> collectEvents() {
		BundleContext context = Activator.getContext();
		Collection<ServiceReference<PdfExportPageEvent>> serviceReferences;
		try {
			serviceReferences = context.getServiceReferences(PdfExportPageEvent.class, null);
		} catch (InvalidSyntaxException e) {
			e.printStackTrace();
			serviceReferences = Collections.emptyList();
		}
		Map<String, PdfExportPageEvent> events = new HashMap<>(serviceReferences.size());
		for (ServiceReference<PdfExportPageEvent> reference : serviceReferences) {
			PdfExportPageEvent event = context.getService(reference);
			String eventName = event.toString();
			events.put(eventName, event);
		}
		return events;
	}

	private void safeInvoke(String eventName, PdfExportPageEvent event, Consumer<PdfExportPageEvent> r) {
		try {
			r.accept(event);
		} catch (Exception e) {
			LOGGER.error("Failed to invoke PdfExportPageEvent::" + eventName + " for " + event.toString(), e);
		}
	}

	@Override
	public void onChapter(PdfWriter writer, Document document, float paragraphPosition, Paragraph title) {
		events.values().forEach(e -> safeInvoke("onChapter", e, ee -> ee.onChapter(writer, document, paragraphPosition, title)));
	}

	@Override
	public void onChapterEnd(PdfWriter writer, Document document, float position) {
		events.values().forEach(e -> safeInvoke("onChapterEnd", e, ee -> ee.onChapterEnd(writer, document, position)));
	}

	@Override
	public void onGenericTag(PdfWriter writer, Document document, Rectangle rect, String text) {
		events.values().forEach(e -> safeInvoke("onGenericTag", e, ee -> ee.onGenericTag(writer, document, rect, text)));
	}

	@Override
	public void onOpenDocument(PdfWriter writer, Document document) {
		events.values().forEach(e -> safeInvoke("onOpenDocument", e, ee -> ee.onOpenDocument(writer, document)));
	}

	@Override
	public void onCloseDocument(PdfWriter writer, Document document) {
		events.values().forEach(e -> safeInvoke("onCloseDocument", e, ee -> ee.onCloseDocument(writer, document)));
	}

	@Override
	public void onParagraph(PdfWriter writer, Document document, float paragraphPosition) {
		events.values().forEach(e -> safeInvoke("onParagraph", e, ee -> ee.onParagraph(writer, document, paragraphPosition)));
	}

	@Override
	public void onParagraphEnd(PdfWriter writer, Document document, float paragraphPosition) {
		events.values().forEach(e -> safeInvoke("onParagraphEnd", e, ee -> ee.onParagraphEnd(writer, document, paragraphPosition)));
	}

	@Override
	public void onSection(PdfWriter writer, Document document, float paragraphPosition, int depth, Paragraph title) {
		events.values().forEach(e -> safeInvoke("onSection", e, ee -> ee.onSection(writer, document, paragraphPosition, depth, title)));
	}

	@Override
	public void onSectionEnd(PdfWriter writer, Document document, float position) {
		events.values().forEach(e -> safeInvoke("onSectionEnd", e, ee -> ee.onSectionEnd(writer, document, position)));
	}

	@Override
	public void onStartPage(PdfWriter writer, Document document) {
		events.values().forEach(e -> safeInvoke("onStartPage", e, ee -> ee.onStartPage(writer, document)));
	}

	@Override
	public void onEndPage(PdfWriter writer, Document document) {
		events.values().forEach(e -> safeInvoke("onEndPage", e, ee -> ee.onEndPage(writer, document)));
	}

}
