/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.viewpoints;

import java.util.Collection;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.browsing.ui.common.NodeContextUtil;
import org.simantics.browsing.ui.content.Viewpoint;

/**
 * @author Tuukka Lehtonen
 */
public abstract class ViewpointStub implements Viewpoint {

    protected NodeContext[] children    = Viewpoint.PENDING_CHILDREN;
    protected Boolean        hasChildren = Viewpoint.PENDING_HAS_CHILDREN;

    final public void setChildren(PrimitiveQueryUpdater updater, NodeContext[] children) {
        if (children == null)
            throw new NullPointerException(this + ": null children produced by " + getClass().getName());
        for(NodeContext c : children) updater.incRef(c);
        for(NodeContext c : this.children) updater.decRef(c);
        this.children = children;
    }

    public final void setHasChildren(Boolean hasChildren) {
        if (children == null)
            throw new NullPointerException(this + ": null hasChildren produced by " + getClass().getName());
        this.hasChildren = hasChildren;
    }

    /**
     * A utility method for transforming an array of objects into an
     * INodeContext array which is the return value of a ViewpointFactory.
     * 
     * <p>
     * The INodeContext's are constructed using
     * {@link NodeContextBuilder#buildWithInput(Object)}.
     * </p>
     * 
     * @param children
     * @return the specified children wrapped into simple <code>INodeContext</code>s
     */
    public NodeContext[] toContextsWithInput(Object... children) {
        return NodeContextUtil.toContextsWithInput(children);
    }

    /**
     * A utility method for transforming a collection of objects into an
     * INodeContext array which is the return value of a ViewpointFactory.
     * 
     * <p>
     * The INodeContext's are constructed using
     * {@link NodeContextBuilder#buildWithInput(Object)}.
     * </p>
     * 
     * @param children
     * @return the specified children wrapped into simple <code>INodeContext</code>s
     */
    public NodeContext[] toContextsWithInput(Collection<?> children) {
        return NodeContextUtil.toContextsWithInput(children);
    }

    /**
     * A utility method for transforming a collection of objects into an
     * INodeContext array which is the return value of a ViewpointFactory.
     * 
     * <p>
     * The INodeContext's are constructed using the specified factory.
     * </p>
     * 
     * @param children
     * @return the specified children wrapped into INodeContext's through the
     *         specified <code>factory</code>
     */
    public NodeContext[] toContexts(Collection<?> children, NodeContextUtil.NodeContextFactory factory) {
        return NodeContextUtil.toContexts(children, factory);
    }

    /**
     * @param <T>
     * @param clazz
     * @return input of the specified class
     * @throws ClassCastException if the input class does not match the
     *         specified class
     * @throws NullPointerException if the input is null
     */
    @SuppressWarnings("unchecked")
    protected <T> T getInput(NodeContext context) {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o == null)
            throw new NullPointerException("null input");
        return (T) o;
    }

    /**
     * @param <T>
     * @param clazz
     * @return <code>null</code> if input is <code>null</code> or if the class
     *         does not match
     */
    @SuppressWarnings("unchecked")
    protected <T> T tryGetInput(NodeContext context, Class<T> clazz) {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o != null && clazz.isInstance(o))
            return (T) o;
        return null;
    }

}
