/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.modifiers;

import java.util.Comparator;

/**
 * @author Tuukka Lehtonen
 * 
 * @param <T>
 */
public class EnumeratedValue<T> implements Comparable<EnumeratedValue<T>>{

    private static class InvalidEnumeratedValue<T> extends EnumeratedValue<T> {
        public InvalidEnumeratedValue(String name) {
            super(name, null);
        }
    }

    public static final <T> EnumeratedValue<T> invalid(String name) {
        return new InvalidEnumeratedValue<T>(name);
    }

    public static final boolean isInvalid(EnumeratedValue<?> value) {
        return value instanceof InvalidEnumeratedValue<?>;
    }

    public static final Comparator<EnumeratedValue<? extends Object>> CASE_INSENSITIVE_NAME_ORDER = new Comparator<EnumeratedValue<? extends Object>>() {
        @Override
        public int compare(EnumeratedValue<? extends Object> o1, EnumeratedValue<? extends Object> o2) {
            return o1.getName().compareToIgnoreCase(o2.getName());
        }
    };

    private final String name;

    private final T      object;

    public static EnumeratedValue<String> make(String name) {
        return new EnumeratedValue<String>(name, name);
    }

    public static <T> EnumeratedValue<T> make(String name, T object) {
        return new EnumeratedValue<T>(name, object);
    }

    public EnumeratedValue(String name, T object) {
        if (name == null)
            throw new NullPointerException("null enumerated value name");
        this.name = name;
        this.object = object;
    }

    /**
     * @return the name of the enumerated value
     */
    public String getName() {
        return name;
    }

    /**
     * @return the object bound to this enumerated value, may be
     *         <code>null</code>
     */
    public T getObject() {
        return object;
    }

    @Override
    public String toString() {
        return name + "[" + object + "]";
    }

    @Override
    public int compareTo(EnumeratedValue<T> o) {
        return getName().compareTo(o.getName());
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + name.hashCode();
        result = prime * result + ((object == null) ? 0 : object.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        EnumeratedValue<?> other = (EnumeratedValue<?>) obj;
        if (!name.equals(other.name))
            return false;
        if (object == null) {
            if (other.object != null)
                return false;
        } else if (!object.equals(other.object))
            return false;
        return true;
    }

}
