/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.annotation.ui.wizard;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Deque;
import java.util.HashMap;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.Simantics;
import org.simantics.annotation.ui.Activator;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.databoard.container.DataFormatException;
import org.simantics.databoard.container.FormatHandler;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.impl.DefaultPasteImportAdvisor;
import org.simantics.db.management.ISessionContext;
import org.simantics.graph.db.TransferableGraphs;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class AnnotationTypeImportWizard extends Wizard implements IImportWizard {

    private static final int MAX_RECENT_IMPORT_PATHS = 10;

    ImportPlan        importModel;

    private boolean readPreferences(IStructuredSelection selection) {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        String recentPathsPref = store.getString(Preferences.RECENT_ANNOTATION_TYPE_IMPORT_LOCATIONS);
        Deque<String> recentImportPaths = Preferences.decodePaths(recentPathsPref);

        ISessionContext ctx = Simantics.getSessionContext();
        if (ctx == null)
            return false;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return false;

        importModel = new ImportPlan(ctx, recentImportPaths);
        importModel.project = project;
        importModel.selection = selection.getFirstElement();

        return true;
    }

    private void writePreferences() throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(Preferences.RECENT_ANNOTATION_TYPE_IMPORT_LOCATIONS, Preferences.encodePaths(importModel.recentLocations));

        if (store.needsSaving())
            store.save();
    }

    public AnnotationTypeImportWizard() {
        setWindowTitle("Import Annotation Type");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences(selection);
    }

    @Override
    public void addPages() {
        super.addPages();
        if (importModel != null) {
            addPage(new AnnotationTypeImportPage(importModel));
        } else {
            addPage(new NoProjectPage("Import Annotation Type"));
        }
    }

    @Override
    public boolean performFinish() {
        try {
        	importModel.recentLocations.addFirst(importModel.importLocation.getAbsolutePath());
            Preferences.removeDuplicates(importModel.recentLocations);
            if (importModel.recentLocations.size() > MAX_RECENT_IMPORT_PATHS)
            	importModel.recentLocations.pollLast();

            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        try {
            getContainer().run(true, true, new IRunnableWithProgress() {
                @Override
                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                    	Resource target = ResourceAdaptionUtils.toSingleResource(importModel.selection);
                        doImport(monitor, importModel.importLocation, importModel.sessionContext.getSession(), target);
                    } catch (Exception e) {
                        throw new InvocationTargetException(e);
                    }
                }
            });
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            WizardPage cp = (WizardPage) getContainer().getCurrentPage();
            if (t instanceof IOException) {
                cp.setErrorMessage("An I/O problem occurred while importing annotation type.\n\nMessage: " + e.getMessage());
            }
            ErrorLogger.defaultLogError(t);
            return false;
        } catch (InterruptedException e) {
            ExceptionUtils.logAndShowError(e);
            return false;
        }

        return true;
    }


    public static void doImport(final IProgressMonitor monitor, File modelFile, final Session session, final Resource target)
    throws IOException, SerializationException, DatabaseException {
        try {
            final DefaultPasteImportAdvisor advisor = new DefaultPasteImportAdvisor(target);

            monitor.beginTask("Loading annotation type from disk", 1000);
            try {
                FormatHandler<Object> handler1 = new FormatHandler<Object>() {
                    @Override
                    public Binding getBinding() {
                        return TransferableGraph1.BINDING;
                    }

                    @Override
                    public Object process(DataContainer container)
                            throws Exception {
                        monitor.worked(100);
                        monitor.setTaskName("Importing annotation type into database");
                        TransferableGraphs.importGraph1(session, (TransferableGraph1)container.content.getValue(), 
                                advisor, null);
                        monitor.worked(850);
                        return null;
                    }
                };

                HashMap<String, FormatHandler<Object>> handlers = new HashMap<String, FormatHandler<Object>>();
                handlers.put(Constants.ANNOTATION_TYPE_FORMAT_V1, handler1);

                try {
                    DataContainers.readFile(modelFile, handlers);
                } catch(DataFormatException e) {
                    throw new IOException(e);
                } catch(IOException e) {
                    throw e;
                } catch(Exception e) {
                    if(e instanceof RuntimeException)
                        throw (RuntimeException)e;
                    else
                        throw new RuntimeException(e);
                }
            } catch(IOException e) {
            }

            monitor.setTaskName("Postprocessing");
            monitor.worked(50);

        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            monitor.done();
        }
    }

}
