/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.set.hash;

import com.koloboke.collect.set.ShortSet;
import com.koloboke.compile.KolobokeSet;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.*;


/**
 * This class consists only of static factory methods to construct {@code HashShortSet}s, and
 * the default {@link HashShortSetFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashShortSet
 * @see KolobokeSet @KolobokeSet
 */
public final class HashShortSets {

    private static class DefaultFactoryHolder {
        private static final HashShortSetFactory defaultFactory =
                ServiceLoader.load(HashShortSetFactory.class).iterator().next();
    }

    /**
     * Returns the default implementation of {@link HashShortSetFactory}, to which all static methods
     * in this class delegate.
     *
     * @return the default implementation of {@link HashShortSetFactory}
    
     * @throws RuntimeException if no implementations of {@link HashShortSetFactory} are provided
     */
    @Nonnull
    public static  HashShortSetFactory getDefaultFactory() {
        return (HashShortSetFactory) DefaultFactoryHolder.defaultFactory;
    }
    
    

    


    /**
     * Constructs a new empty mutable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet() newMutableSet()}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
    public static  HashShortSet newMutableSet() {
        return getDefaultFactory().newMutableSet();
    }

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(int) newMutableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
    public static  HashShortSet newMutableSet(int expectedSize) {
        return getDefaultFactory().newMutableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterable, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterator, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterator<Short> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Consumer, int) newMutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashShortSet newMutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * short[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashShortSet newMutableSet(@Nonnull short[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Short[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashShortSet newMutableSet(@Nonnull Short[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterable) newMutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterable, Iterable) newMutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2) {
        return getDefaultFactory().newMutableSet(elems1, elems2);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Iterator) newMutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashShortSet newMutableSet(
            @Nonnull Iterator<Short> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Consumer) newMutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashShortSet newMutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newMutableSet(elementsSupplier);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * short[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashShortSet newMutableSet(@Nonnull short[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSet(
     * Short[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashShortSet newMutableSet(@Nonnull Short[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSetOf(
     * short) newMutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
    public static  HashShortSet newMutableSetOf(short e1) {
        return getDefaultFactory().newMutableSetOf(e1);
    }

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSetOf(
     * short, short) newMutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
    public static  HashShortSet newMutableSetOf(short e1, short e2) {
        return getDefaultFactory().newMutableSetOf(e1, e2);
    }

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSetOf(
     * short, short, short) newMutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
    public static  HashShortSet newMutableSetOf(short e1, short e2, short e3) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSetOf(
     * short, short, short, short) newMutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
    public static  HashShortSet newMutableSetOf(short e1, short e2, short e3, short e4) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newMutableSetOf(short, short, short,
     * short, short, short...) newMutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
    public static  HashShortSet newMutableSetOf(short e1, short e2, short e3, short e4,
            short e5, short... restElements) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    /**
     * Constructs a new empty updatable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet() newUpdatableSet()}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet() {
        return getDefaultFactory().newUpdatableSet();
    }

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(int) newUpdatableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(int expectedSize) {
        return getDefaultFactory().newUpdatableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterable, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterator, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterator<Short> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Consumer, int) newUpdatableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * short[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(@Nonnull short[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Short[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashShortSet newUpdatableSet(@Nonnull Short[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterable) newUpdatableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterable, Iterable) newUpdatableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Iterator) newUpdatableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(
            @Nonnull Iterator<Short> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Consumer) newUpdatableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * short[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashShortSet newUpdatableSet(@Nonnull short[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSet(
     * Short[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashShortSet newUpdatableSet(@Nonnull Short[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSetOf(
     * short) newUpdatableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
    public static  HashShortSet newUpdatableSetOf(short e1) {
        return getDefaultFactory().newUpdatableSetOf(e1);
    }

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSetOf(
     * short, short) newUpdatableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
    public static  HashShortSet newUpdatableSetOf(short e1, short e2) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2);
    }

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSetOf(
     * short, short, short) newUpdatableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
    public static  HashShortSet newUpdatableSetOf(short e1, short e2, short e3) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSetOf(
     * short, short, short, short) newUpdatableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
    public static  HashShortSet newUpdatableSetOf(short e1, short e2, short e3, short e4) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newUpdatableSetOf(short, short, short,
     * short, short, short...) newUpdatableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
    public static  HashShortSet newUpdatableSetOf(short e1, short e2, short e3, short e4,
            short e5, short... restElements) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4, e5, restElements);
    }


    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterable, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterator, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterator<Short> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Consumer, int) newImmutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * short[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(@Nonnull short[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Short[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashShortSet newImmutableSet(@Nonnull Short[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterable) newImmutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterable, Iterable) newImmutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2) {
        return getDefaultFactory().newImmutableSet(elems1, elems2);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Iterator) newImmutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(
            @Nonnull Iterator<Short> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Consumer) newImmutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newImmutableSet(elementsSupplier);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * short[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashShortSet newImmutableSet(@Nonnull short[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSet(
     * Short[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashShortSet newImmutableSet(@Nonnull Short[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSetOf(
     * short) newImmutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
    public static  HashShortSet newImmutableSetOf(short e1) {
        return getDefaultFactory().newImmutableSetOf(e1);
    }

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSetOf(
     * short, short) newImmutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
    public static  HashShortSet newImmutableSetOf(short e1, short e2) {
        return getDefaultFactory().newImmutableSetOf(e1, e2);
    }

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSetOf(
     * short, short, short) newImmutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
    public static  HashShortSet newImmutableSetOf(short e1, short e2, short e3) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSetOf(
     * short, short, short, short) newImmutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
    public static  HashShortSet newImmutableSetOf(short e1, short e2, short e3, short e4) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashShortSetFactory#newImmutableSetOf(short, short, short,
     * short, short, short...) newImmutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
    public static  HashShortSet newImmutableSetOf(short e1, short e2, short e3, short e4,
            short e5, short... restElements) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    private HashShortSets() {}
}
