/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.annotations.Length;
import org.simantics.databoard.annotations.MIMEType;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.annotations.Pattern;
import org.simantics.databoard.annotations.Range;
import org.simantics.databoard.annotations.Union;
import org.simantics.databoard.annotations.Unit;
import org.simantics.databoard.parser.repository.DataTypeRepository;
import org.simantics.databoard.parser.repository.DataTypeSyntaxError;
import org.simantics.databoard.type.Component;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.RecordType;


public class ParserSuccessTests extends TestCase {

	DataTypeRepository repository;
	Map<String, Datatype> reference;
	
	@Before
	public void setUp() {
		repository = new DataTypeRepository();
		reference = new HashMap<String, Datatype>();
	}	
	
	public void printRepository() {
		for(String name : repository.getTypeNames()) {
			Datatype type = repository.get(name);
			System.out.println(name + " = " + type + " ("
					+ type.getClass() + ")");
		}	
	}
	
	@After
	public void compare() {
		for(String name : reference.keySet())
			if(!repository.getTypeNames().contains(name))
				fail("Type " + name + " is not in repository.");
		for(String name : repository.getTypeNames())
			if(!reference.containsKey(name))
				fail("Type " + name + " should not be in the repository.");
		for(String name : repository.getTypeNames())
			assertEquals("Problem with type " + name + ".",
					reference.get(name), repository.get(name));
	}		

	@Test
	public void testBuiltins() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type MyBoolean = Boolean " +
				"type MyByte = Byte " +
				"type MyInteger = Integer " + 				
				"type MyLong = Long " +
				"type MyFloat = Float " +
				"type MyDouble = Double " +
				"type MyString = String "
				);
		reference.put("MyBoolean", Datatypes.getDatatypeUnchecked(Boolean.class));
		reference.put("MyByte", Datatypes.getDatatypeUnchecked(Byte.class));
		reference.put("MyInteger", Datatypes.getDatatypeUnchecked(Integer.class));
		reference.put("MyFloat", Datatypes.getDatatypeUnchecked(Float.class));
		reference.put("MyDouble", Datatypes.getDatatypeUnchecked(Double.class));
		reference.put("MyLong", Datatypes.getDatatypeUnchecked(Long.class));
		reference.put("MyString", Datatypes.getDatatypeUnchecked(String.class));
	}
	
	@Test
	public void testArray() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Array1 = Integer[] " +
				"type Array2 = String[][] " 
				);
		reference.put("Array1", Datatypes.getDatatypeUnchecked(int[].class));		
		reference.put("Array2", Datatypes.getDatatypeUnchecked(String[][].class));
	}
	
	static class Optional1 {
		@Optional Integer a;
		@Optional float[] b;
	}
	
	@Test
	public void testOptional() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Optional1 = {" +
				"a : Optional(Integer)," +
				"b : Optional(Float[])" +
				"}"
				);
		reference.put("Optional1", Datatypes.getDatatypeUnchecked(Optional1.class));		
	}
	
	static class Record1 {		
	}
	
	static class Record2 {	
		String a;
		Integer b;
	}
	
	@Test
	public void testRecord() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Record1 = {} " +
				"type Record2 = { a : String, b : Integer } " 
				);
		reference.put("Record1", Datatypes.getDatatypeUnchecked(Record1.class));		
		reference.put("Record2", Datatypes.getDatatypeUnchecked(Record2.class));
	}
	
	public Datatype tuple(Datatype ... types) {
		RecordType tuple = new RecordType();
		Component[] components = new Component[types.length];
		for(int i=0;i<types.length;++i)
			components[i] = new Component(Integer.toString(i), types[i]);
		tuple.setComponents(components);
		return tuple;
	}
	
	@Test
	public void testTuple() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Tuple1 = () " +
				"type Tuple2 = (Integer, String) " +
				"type NotTuple = (Double) "				
				);
		reference.put("Tuple1", Datatypes.getDatatypeUnchecked(Record1.class));
		reference.put("Tuple2", tuple(
				Datatypes.getDatatypeUnchecked(Integer.class),
				Datatypes.getDatatypeUnchecked(String.class)
				));
		reference.put("NotTuple", Datatypes.getDatatypeUnchecked(Double.class));	
	}
	
	@Union({A.class, /*B.class,*/ C.class})
	interface Union1 {		
	}
	
	static class A implements Union1 {}
	static class B implements Union1 { int value; }
	static class C implements Union1 { float x; int y; }
	
	@Test
	public void testUnion() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Union1 = | A " +
				"/*             | B Integer */" +
				"             | C { x : Float, y : Integer } " 
				);
		reference.put("Union1", Datatypes.getDatatypeUnchecked(Union1.class));
	}
	
	@Test
	public void testReferences() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Ref1 = String " +
				"type Ref2 = { a : Ref1, b : Ref3 } " +
				"type Ref3 = Ref4 " + 
				"type Ref4 = Integer "				 
				);
		reference.put("Ref1", Datatypes.getDatatypeUnchecked(String.class));
		reference.put("Ref2", Datatypes.getDatatypeUnchecked(Record2.class));
		reference.put("Ref3", Datatypes.getDatatypeUnchecked(Integer.class));
		reference.put("Ref4", Datatypes.getDatatypeUnchecked(Integer.class));		
	}
	
	static class Rec1 {
		Rec2[] ref1;
	}
	
	static class Rec2 {
		@Optional Rec1 ref2;
	}
	
	@Test
	public void testRecursion() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type Rec1 = { ref1 : Rec2[] } " + 
				"type Rec2 = { ref2 : Optional(Rec1) } "
				);
		reference.put("Rec1", Datatypes.getDatatypeUnchecked(Rec1.class));
		reference.put("Rec2", Datatypes.getDatatypeUnchecked(Rec2.class));		
	}
	
	static class BuiltinAnnotations {
		@Range("[-9..10]") @Unit("foo") int a;
		@Range("[0..3453453453345]") @Unit("foo") long b;
		@Range("[-4.3..10]") @Unit("foo") float c;
		@Range("[0..10.3]") @Unit("foo") double d;
		@Length("0") @MIMEType("foo") @Pattern("\\S{2,9}") String e;
	}
	
	@Test
	public void testBuiltinAnnotations() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type BuiltinAnnotations = {" +
				"a : Integer(range=[-9..10], unit=\"foo\")," +
				"b : Long(range=[0..3453453453345], unit=\"foo\")," +	
				"c : Float(range=[-4.3..10], unit=\"foo\")," +	
				"d : Double(range=[0..10.3], unit=\"foo\")," +	
				"e : String(length=[0..10], mimeType=\"foo\", pattern=\"\\\\S{2,9}\")" +	
				"}"
				);
		reference.put("BuiltinAnnotations", 
				Datatypes.getDatatypeUnchecked(BuiltinAnnotations.class));		
	}
	
	static class ArrayAnnotations {
		@Length("[2..]") float[] a;		
		@Length("[2..]") double[] b;		
		@Length("[1..3]") String[] c;
		@Length("2") byte[] d;
	}
	
	@Test
	public void testArrayAnnotations() throws DataTypeSyntaxError {
		repository.addDefinitions(
				"type ArrayAnnotations = {" +
				"a : Float[2..]," +
				"b : Double[..2]," +
				"c : String[1..3]," +
				"d : Byte[2]" +
				"}"
				);
		reference.put("ArrayAnnotations", 
				Datatypes.getDatatypeUnchecked(ArrayAnnotations.class));		
	}
}
