/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures.collections;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.utils.datastructures.Pair;

/**
 * @author Tuukka Lehtonen
 */
public final class CollectionUtils {

    /**
     * Add the elements in <code>toBeAdded</code> to addTo. The only difference
     * to {@link Collection#addAll(Collection)} is that <code>toBeAdded</code>
     * may be null.
     * 
     * @param <T>
     * @param toBeAdded collection of elements to be added to <code>addTo</code> or
     *        <code>null</code> to do nothing
     * @param addTo the collection the elements should be added to
     * @return <code>true</code> if the destination collection changed as a
     *         result of the operation
     */
    public static <T> boolean checkedAdd(Collection<T> toBeAdded, Collection<T> addTo) {
        assert addTo != null;
        if (toBeAdded == null)
            return false;
        return addTo.addAll(toBeAdded);
    }

    /**
     * @return the accumulated size of all the specified collections
     */
    public static int sizeAll(Collection<?>... collections) {
        int size = 0;
        for (Collection<?> c : collections)
            size += c.size();
        return size;
    }

    /**
     * @return a new ArrayList<T> that sequentially contains all the elements in
     *         the specified collections
     */
    public static <T> Collection<T> join(@SuppressWarnings("unchecked") Collection<T>... collections) {
        int size = sizeAll(collections);
        ArrayList<T> result = new ArrayList<T>(size);
        for (Collection<T> c : collections)
            result.addAll(c);
        return result;
    }

    /**
     * @return a new HashSet<T> that contains once all the elements in the
     *         specified collections
     */
    public static <T> Set<T> join(@SuppressWarnings("unchecked") Set<T>... collections) {
        int size = sizeAll(collections);
        HashSet<T> result = new HashSet<T>(size);
        for (Collection<T> c : collections)
            result.addAll(c);
        return result;
    }

    /**
     * A specialization of {@link #join(Collection...)} for two arguments.
     */
    public static <T> Collection<T> join(Collection<T> first, Collection<T> second) {
        int size = first.size();
        size += second.size();
        ArrayList<T> result = new ArrayList<T>(size);
        result.addAll(first);
        result.addAll(second);
        return result;
    }
    
    static class PairFirstComparator<K extends Comparable<K>> implements Comparator<Pair<K, ?>> {

		@Override
		public int compare(Pair<K,?> o1, Pair<K,?> o2) {
			return o1.first.compareTo(o2.first);
		}
    	
    }
    
    static class PairSecondComparator<V extends Comparable<V>> implements Comparator<Pair<?, V>> {

 		@Override
 		public int compare(Pair<?, V> o1, Pair<?, V> o2) {
 			return o1.second.compareTo(o2.second);
 		}
     	
     }    
    
    public static <K> Collection<Pair<K,Integer>> valueSortedEntries(TObjectIntHashMap<K> map) {
    	
    	ArrayList<Pair<K,Integer>> result = new ArrayList<Pair<K,Integer>>();
    	for(K key : map.keySet()) result.add(new Pair<K,Integer>(key, map.get(key)));
    	Collections.sort(result, new PairSecondComparator<Integer>());
    	return result;
    	
    }

    public static <K, V extends Comparable<V>> List<Pair<K,V>> valueSortedEntries(Map<K,V> map) {
    	
    	ArrayList<Pair<K,V>> result = new ArrayList<Pair<K,V>>(); 
    	for(Map.Entry<K, V> entry : map.entrySet()) result.add(new Pair<K,V>(entry.getKey(), entry.getValue()));
    	Collections.sort(result, new PairSecondComparator<V>());
    	return result;
    	
    }
    
    public static <K extends Comparable<K>, V> List<V> sortedBy(List<K> keys, List<V> values) {

    	ArrayList<Pair<V,K>> work = new ArrayList<Pair<V,K>>();
    	for(int i=0;i<keys.size();i++) work.add(Pair.make(values.get(i), keys.get(i)));
    	Collections.sort(work, new PairSecondComparator<K>());
    	ArrayList<V> result = new ArrayList<V>();
    	for(Pair<V, K> pair : work) result.add(pair.first);
    	return result;

    }

    public static <K extends Comparable<K>,V> Collection<V> sortByFirst(List<Pair<K,V>> collection) {
    	Collections.sort(collection, new PairFirstComparator<K>());
    	ArrayList<V> values = new ArrayList<V>();
    	for(Pair<K,V> pair : collection) values.add(pair.second);
    	return values;
    }
    
    /**
     * A specialization of {@link #join(Set...)} for two arguments.
     */
    public static <T> Set<T> join(Set<T> first, Set<T> second) {
        int size = first.size();
        size += second.size();
        HashSet<T> result = new HashSet<T>(size);
        result.addAll(first);
        result.addAll(second);
        return result;
    }

    /*
     * Some tests.
     */
    @SuppressWarnings({ "unused", "unchecked" })
    public static void main(String[] args) {
        Collection<String> c1 = Collections.singleton("FOO");
        Collection<String> c2 = Collections.singleton("BAR");
        Collection<String> c3 = Collections.singleton("BAZ");
        Collection<String> c4 = Collections.singleton("FOO");

        Collection<String> c123 = join(c1, c2, c3);
        Collection<String> c12 = join(c1, c2);

        Set<String> s1 = Collections.singleton("FOO");
        Set<String> s2 = Collections.singleton("BAR");
        Set<String> s3 = Collections.singleton("BAZ");
        Set<String> s4 = Collections.singleton("FOO");

        Set<String> s123 = join(s1, s2, s3);
        Set<String> s12 = join(s1, s2);
        Set<String> s14 = join(s1, s4);
    }

    public static <T> void toggle(Set<T> src, Set<T> toggleSet)
    {
    	for (T i : toggleSet)
    	{
    		if (src.contains(i))
    			src.remove(i);
    		else 
    			src.add(i);
    	}
    }

    public static <T> List<T> toList(@SuppressWarnings("unchecked") T...objs) 
    {
    	ArrayList<T> result = new ArrayList<T>(objs.length);
    	for (T o : objs)
    		result.add(o);
    	return result;
    }

    public static <T> Set<T> toSet(@SuppressWarnings("unchecked") T...objs)
    {
    	THashSet<T> result = new THashSet<T>(objs.length);
    	for (T o : objs)
    		result.add(o);
    	result.compact();
    	return result;
    }
    
    /**
     * Remove elements that appear more than once. Keep order otherwise.
     * @param list to be pruned
     */
    public static void unique(List<?> list) { 
    	int i = list.size()-1;
    	while (i>0) {
    		Object o = list.get(i);
    		int index = list.indexOf(o);
    		if (index>=0 && index < i) {
    			list.remove(index);
    			i--;
    		} else {
    			i--;
    		}
    	}
    }

    public static <T> T element(Collection<T> collection, int n) {
    	
        if (collection instanceof List<?>)
            return ((List<T>) collection).get(n);
    	if(n >= collection.size()) throw new IllegalArgumentException();
    	Iterator<T> it = collection.iterator();
    	for( int i=0;i<n-1;i++ ) it.next();
    	return it.next();
    	
    }
    
    public static String toString(Collection<?> collection, String separator) {
    	StringBuilder sb = new StringBuilder();
    	
    	int index = 0;
    	for (Object o : collection) {
    		if ( index++>0 ) sb.append(separator);
    		sb.append( o.toString() );
    	}
    	
    	return sb.toString();
    }
    
    public static String toString(Object[] array, String separator) {
    	StringBuilder sb = new StringBuilder();
    	
    	int index = 0;
    	for (Object o : array) {
    		if ( index++>0 ) sb.append(separator);
    		sb.append( o.toString() );
    	}
    	
    	return sb.toString();
    }
    
}
