/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.nodes;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Iterator;

import org.simantics.scenegraph.IDynamicSelectionPainterNode;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DNode;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.events.EventTypes;
import org.simantics.scenegraph.g2d.events.KeyEvent.KeyPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseClickEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDoubleClickedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.g2d.snap.ISnapAdvisor;
import org.simantics.scenegraph.utils.GeometryUtils;
import org.simantics.scenegraph.utils.NodeUtil;

public class PathNode extends G2DNode implements IDynamicSelectionPainterNode {

    private static final long serialVersionUID = 8508750881358776559L;

    private static final Double PICK_DISTANCE = 5.0;
    private BasicStroke stroke;

    private Path2D path = new Path2D.Double();
    private boolean dirtyPath = true;
    private ArrayList<Point2D> points;
    private IPathListener pathListener;
    private Point2D dragKeyPoint = null;
    private Point2D selectedKeyPoint = null;
    private Point2D hoverKeyPoint = null;
    private Point2D pressKeyPoint = null;
    private Integer pressSegment = null;
    private double mouseX;
    private double mouseY;
    private boolean closed;
    private boolean align = false;
    private Color fill;
    private Color color;

    public void init(ArrayList<Point2D> points, boolean closed) {
        // we don't want to lose the editing state due to changes echoed by DB
        if (!points.equals(this.points) || closed != this.closed) {
            this.points = points;
            this.closed = closed;
            if (points.isEmpty()) {
                points.add(new Point2D.Double(0, 0));
            }
            if (points.size() == 1) {
                this.closed = false;
            }
            resetEditingState();
            dirtyPath = true;
            repaint();
        }
    }

    public void setStyle(PathNodeStyle style) {
        float[] color = style.getColor();
        this.color = new Color(color[0], color[1], color[2], color[3]);
        float[] fill = style.getFillColor();
        this.fill = new Color(fill[0], fill[1], fill[2], fill[3]);
        stroke = new BasicStroke(style.getStrokeWidth(), BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER, 5.0f, null, 0.0f);
        repaint();
    }

    private void resetEditingState() {
        dragKeyPoint = null;
        selectedKeyPoint = null;
        hoverKeyPoint = null;
        pressKeyPoint = null;
        pressSegment = null;
    }

    public void setPathListener(IPathListener pathListener) {
        this.pathListener = pathListener;
    }

    private void flushChanges() {
        if (pathListener != null) {
            pathListener.pathChanged(points, closed);
        }
    }

    public boolean isClosed() {
        return closed;
    }

    public Shape getShape() {
        return path;
    }

    private void updatePath() {
        dirtyPath = false;
        path.reset();
        Iterator<Point2D> it = points.iterator();
        if (!it.hasNext()) {
            path.moveTo(0.0, 0.0);
            return;
        }
        Point2D first = it.next();
        path.moveTo(first.getX(), first.getY());
        while (it.hasNext()) {
            Point2D lineTo = it.next();
            path.lineTo(lineTo.getX(), lineTo.getY());
        }
        if (closed) path.closePath();
    }

    private double calculateScale() {
        NavigationNode nn = NodeUtil.findNearestParentNode(this, NavigationNode.class);
        double scale = 1.0;
        if (nn != null) {
            AffineTransform copy = (AffineTransform)transform.clone();
            copy.concatenate(nn.getTransform());
            scale = GeometryUtils.getScale(copy);
        }
        return scale;
    }

    private void resolveTransformation(INode n, AffineTransform T) {
        INode p = n.getParent();
        if (!(p instanceof NavigationNode) && p != null) {
            resolveTransformation(p, T);
        }
        if (n instanceof G2DParentNode) {
            T.concatenate(((G2DParentNode)n).getTransform());
        }
    }

    private Point2D snap(Point2D point) {
        GridNode grid = lookupNode(GridNode.GRID_NODE_ID, GridNode.class);
        if (align) {
            Point2D ref = null;
            if (selectedKeyPoint != null) {
                ref = selectedKeyPoint;
            } else if (points.size() > 1) {
                if (dragKeyPoint == points.get(0) && !closed) {
                    ref = points.get(1);
                } else if (dragKeyPoint == points.get(points.size() - 1) && !closed) {
                    ref = points.get(points.size() - 2);
                } else if (points.size() > 2) {
                    int i = points.indexOf(dragKeyPoint);
                    point = snap2(point, points.get(Math.floorMod(i - 1, points.size())), points.get((i + 1) % points.size()));
                }
            }
            if (ref != null) {
                double angle = Math.atan2(point.getY() - ref.getY(), point.getX() - ref.getX());
                double distance = ref.distance(point);
                angle = Math.round(angle / (Math.PI/4)) * (Math.PI/4);
                point.setLocation(ref.getX() + distance * Math.cos(angle), ref.getY() + distance * Math.sin(angle));
            }
        }
        if (grid != null) {
            ISnapAdvisor snapAdvisor = grid.getSnapAdvisor();
            AffineTransform t = transform;
            resolveTransformation(this, transform);
            Point2D p = t.transform(point, null);
            snapAdvisor.snap(p);
            try {
                return t.inverseTransform(p, null);
            } catch (NoninvertibleTransformException e) {
                return point;
            }
        } else {
            return point;
        }
    }

    private Point2D snap2(Point2D point, Point2D ref1, Point2D ref2) {
        double minDistanceSq = Double.MAX_VALUE;
        Point2D result = new Point2D.Double();
        for (int a1 = 0; a1 < 8; a1++) {
            for (int a2 = 0; a2 < 8; a2++) {
                if (a1 % 4 != a2 % 4) {
                    double x1 = ref1.getX();
                    double y1 = ref1.getY();
                    double x2 = x1 + Math.cos(((double)a1) * Math.PI / 4.0);
                    double y2 = y1 + Math.sin(((double)a1) * Math.PI / 4.0);
                    double x3 = ref2.getX();
                    double y3 = ref2.getY();
                    double x4 = x3 + Math.cos(((double)a2) * Math.PI / 4.0);
                    double y4 = y3 + Math.sin(((double)a2) * Math.PI / 4.0);

                    double denom = (x1 - x2) * (y3 - y4) - (y1 - y2) * (x3 - x4);
                    double t = ((x1 - x3) * (y3 - y4) - (y1 - y3) * (x3 - x4)) / denom;
                    double x = x1 + t * (x2 - x1);
                    double y = y1 + t * (y2 - y1);
                    double distanceSq = point.distanceSq(x, y);
                    if (distanceSq < minDistanceSq) {
                        minDistanceSq = distanceSq;
                        result.setLocation(x, y);
                    }
                }
            }
        }
        return result;
    }

    @Override
    public void render(Graphics2D g2d) {
        if (dirtyPath) {
            updatePath();
        }
        AffineTransform ot = g2d.getTransform();
        g2d.transform(transform);
        g2d.setStroke(stroke);
        if (closed) {
            g2d.setColor(fill);
            g2d.fill(path);
        }
        g2d.setColor(color);
        g2d.draw(path);

        if (selectedKeyPoint != null) {
            g2d.setColor(Color.RED);
            Path2D path = new Path2D.Double();
            path.moveTo(selectedKeyPoint.getX(), selectedKeyPoint.getY());
            Point2D mouse = snap(new Point2D.Double(mouseX, mouseY));
            path.lineTo(mouse.getX(), mouse.getY());
            g2d.draw(path);
        }

        if (NodeUtil.isSelected(this, 1) || points.size() == 1) {
            drawKeyPoints(g2d, path, points.size() > 1 || NodeUtil.isSelected(this, 1) ? Color.BLUE : Color.BLACK);
        }
        g2d.setTransform(ot);
    }

    private Point2D pickKeyPoint(double x, double y, double tolerance) {
        Point2D best = null;
        double minDistanceSq = Double.MAX_VALUE;
        double toleranceSq = tolerance * tolerance;
        for (int i = 0; i < points.size(); i++) {
            Point2D point = points.get(i);
            double distanceSq = point.distanceSq(x, y);

            if ((distanceSq < toleranceSq) && (distanceSq < minDistanceSq)) {
                minDistanceSq = distanceSq;
                best = point;
            }
        }
        return best;
    }

    public boolean intersectsOutline(AffineTransform t, Rectangle2D rect) {
        if (points.size() > 1) {
            for (int i = 0; i < points.size() - (closed ? 0 : 1); i++) {
                Point2D point1 = points.get(i);
                Point2D point2 = points.get((i + 1) % points.size());

                Line2D line = new Line2D.Double(t.transform(point1, null), t.transform(point2, null));
                if (line.intersects(rect)) return true;
            }
        } else {
            Point2D point1 = t.transform(points.get(0), null);
            return rect.contains(point1);
        }
        return false;
    }

    private Integer pickSegment(double x, double y, double tolerance) {
        Integer best = null;
        double minDistanceSq = Double.MAX_VALUE;
        double toleranceSq = tolerance * tolerance;
        for (int i = 0; i < points.size() - (closed ? 0 : 1); i++) {
            Point2D point1 = points.get(i);
            Point2D point2 = points.get((i + 1) % points.size());
            double distanceSq = Line2D.ptSegDistSq(point1.getX(), point1.getY(), point2.getX(), point2.getY(), x, y);

            if ((distanceSq < toleranceSq) && (distanceSq < minDistanceSq)) {
                minDistanceSq = distanceSq;
                best = i;
            }
        }
        return best;
    }

    private Point2D addKeyPoint(int pos, double x, double y) {
        Point2D point = snap(new Point2D.Double(x, y));
        points.add(pos, point);
        dirtyPath = true;
        flushChanges();
        return point;
    }

    private void removeKeyPoint(Path2D path, Point2D point) {
        if (points.remove(point)) {
            if (closed && points.size() == 1) {
                closed = false;
            }
            dirtyPath = true;
            flushChanges();
        }
    }

    private void moveKeyPoint(Point2D point, double x, double y) {
        Point2D target = snap(new Point2D.Double(x, y));
        point.setLocation(target.getX(), target.getY());
        dirtyPath = true;
    }

    private void drawKeyPoints(Graphics2D g2d, Path2D path, Color color) {
        double scale = calculateScale();
        double r = PICK_DISTANCE / scale;
        g2d.setStroke(new BasicStroke((float)(2.0 / scale), BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER, 10.0f, null, 0.0f));

        for (Point2D point : points) {
            g2d.setColor(point == selectedKeyPoint ? Color.RED : color);
            Ellipse2D e = new Ellipse2D.Double(point.getX() - r, point.getY() - r, r * 2, r * 2);
            g2d.draw(e);
            if (hoverKeyPoint == point) {
                g2d.fill(e);
            }
        }
    }

    @Override
    public Rectangle2D getBoundsInLocal() {
        if (dirtyPath) updatePath();
        if(path == null) return null;
        return path.getBounds2D();
    }

    @Override
    protected boolean mouseDoubleClicked(MouseDoubleClickedEvent event) {
        if (event.button != MouseClickEvent.LEFT_BUTTON)
            return false;
        Point2D localPos = getMousePosition(event);
        double x = localPos.getX();
        double y = localPos.getY();

        Point2D point = pickKeyPoint(x, y, PICK_DISTANCE / calculateScale());
        if (point != null) {
            if (points.size() > 1) {
                if (point == selectedKeyPoint) {
                    selectedKeyPoint = null;
                }
                removeKeyPoint(path, point);
                repaint();
            }
            return true;
        }

        return false;
    }

    private Point2D getPosition(Point2D controlPosition) {
        Point2D localPos = NodeUtil.worldToLocal(this, controlPosition, new Point2D.Double());
        Point2D result = new Point2D.Double();
        try {
            transform.createInverse().transform(localPos, result);
        } catch (NoninvertibleTransformException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        return result;
    }

    private Point2D getMousePosition(MouseEvent e) {
        return getPosition(e.controlPosition);
    }

    @Override
    protected boolean mouseDragged(MouseDragBegin event) {
        if (!NodeUtil.isSelected(this, 1)) return false;

        if (selectedKeyPoint != null) return true;
        if (event.button != MouseClickEvent.LEFT_BUTTON) return false;

        if (event.button == MouseClickEvent.LEFT_BUTTON && (selectedKeyPoint != null))
            return true;

        Point2D localPos = getMousePosition(event);
        double x = localPos.getX();
        double y = localPos.getY();

        dragKeyPoint = pickKeyPoint(x, y, PICK_DISTANCE / calculateScale());
        if (dragKeyPoint != null) return true;

        return false;
    }

    @Override
    protected boolean mouseMoved(MouseMovedEvent event) {
        if (!NodeUtil.isSelected(this, 1)) return false;
        align = event.isControlDown();
        if (dragKeyPoint != null) {
            Point2D localPos = snap(getMousePosition(event));
            double x = localPos.getX();
            double y = localPos.getY();
            moveKeyPoint(dragKeyPoint, x, y);
            repaint();
            return true;
        }
        Point2D localPos = getMousePosition(event);
        mouseX = localPos.getX();
        mouseY = localPos.getY();
        Point2D pick = pickKeyPoint(mouseX, mouseY, PICK_DISTANCE / calculateScale());
        if (selectedKeyPoint == null || pick == points.get(0) || pick == points.get(points.size() - 1)) {
            if (hoverKeyPoint != pick) {
                hoverKeyPoint = pick;
                repaint();
            }
        } else {
            if (hoverKeyPoint != null) {
                hoverKeyPoint = null;
                repaint();
            }
        }
        if (selectedKeyPoint != null) {
            repaint();
        }
        return false;
    }

    @Override
    protected boolean mouseButtonPressed(MouseButtonPressedEvent event) {
        Point2D localPos = getMousePosition(event);
        double x = localPos.getX();
        double y = localPos.getY();

        pressKeyPoint = pickKeyPoint(x, y, PICK_DISTANCE / calculateScale());
        pressSegment = pickSegment(x, y, PICK_DISTANCE / calculateScale());

        return pressKeyPoint != null || pressSegment != null;
    }

    @Override
    protected boolean mouseButtonReleased(MouseButtonReleasedEvent event) {
        if (!NodeUtil.isSelected(this, 1)) return false;
        if (dragKeyPoint != null) {
            dragKeyPoint = null;
            flushChanges();
            return true;
        }
        if (event.button == MouseClickEvent.RIGHT_BUTTON) {
            if (selectedKeyPoint != null) {
                selectedKeyPoint = null;
                repaint();
            }
            return false;
        }
        if (event.button == MouseClickEvent.MIDDLE_BUTTON) {
            return false;
        }

        Point2D localPos = getMousePosition(event);
        double x = localPos.getX();
        double y = localPos.getY();

        Point2D pick = pickKeyPoint(x, y, PICK_DISTANCE / calculateScale());
        if (pick != null && pick == selectedKeyPoint) {
            selectedKeyPoint = null;
            repaint();

            return true;
        }
        if (pick != null && (pick == points.get(0) || pick == points.get(points.size() - 1)) && pick != selectedKeyPoint && selectedKeyPoint != null) {
            // close path
            selectedKeyPoint = null;
            closed = true;
            flushChanges();
            dirtyPath = true;
            repaint();
            return true;
        } else if (!closed && pick != null && (pick == points.get(0) || pick == points.get(points.size() - 1))) {
            // select first or last keypoint
            selectedKeyPoint = pick;
            repaint();
            return true;
        } else if (selectedKeyPoint != null) {
            if (selectedKeyPoint == points.get(points.size() - 1)) {
                selectedKeyPoint = addKeyPoint(points.size(), x, y);
            } else {
                selectedKeyPoint = addKeyPoint(0, x, y);
            }
            repaint();
            return true;
        } else if (pick == null && pressKeyPoint == null) {
            Integer segment = pickSegment(x, y, PICK_DISTANCE / calculateScale());
            if (segment != null && segment == pressSegment) {
                addKeyPoint(segment + 1, x, y);
            }
        }
        return false;
    }

    @Override
    protected boolean mouseClicked(MouseClickEvent event) {
        if (!NodeUtil.isSelected(this, 1)) return false;
        return selectedKeyPoint != null || dragKeyPoint != null || pressKeyPoint != null || pressSegment != null ;
    }

    @Override
    protected boolean keyPressed(KeyPressedEvent e) {
        if (!NodeUtil.isSelected(this, 1)) return false;

        if (e.keyCode == KeyEvent.VK_ESCAPE) {
            if (dragKeyPoint != null || selectedKeyPoint != null) {
                if (dragKeyPoint != null) {
                    dragKeyPoint = null;
                    flushChanges();
                }
                if (selectedKeyPoint != null) {
                    selectedKeyPoint = null;
                    repaint();
                }
                return true;
            }
        }
        return false;
    }

    @Override
    public int getEventMask() {
        return EventTypes.KeyPressedMask | EventTypes.MouseMovedMask | EventTypes.MouseButtonPressedMask | EventTypes.MouseButtonReleasedMask
                | EventTypes.MouseClickMask | EventTypes.MouseDoubleClickMask | EventTypes.MouseDragBeginMask;
    }

    @Override
    public void init() {
        super.init();
        addEventHandler(this);
    }

    @Override
    public void cleanup() {
        removeEventHandler(this);
        super.cleanup();
    }

    @Override
    public boolean showsSelection() {
        return true;
    }

}
