/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model;

import org.eclipse.core.runtime.IAdaptable;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.AdaptableHintContext;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleVariable;
import org.simantics.db.layer0.request.PossibleVariableRepresents;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.ui.selection.AnyResource;
import org.simantics.ui.selection.AnyVariable;
import org.simantics.ui.selection.ExplorerColumnContentType;
import org.simantics.ui.selection.ExplorerInputContentType;
import org.simantics.ui.selection.WorkbenchSelectionContentType;
import org.simantics.ui.selection.WorkbenchSelectionElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardWorkbenchSelectionElement extends AdaptableHintContext {

	private static final Logger LOGGER = LoggerFactory.getLogger(StandardWorkbenchSelectionElement.class);
	
	public static WorkbenchSelectionElement nodeContextToWorkbenchSelectionElement (NodeContext context) {
		return (new StandardWorkbenchSelectionElement(context));
	}
	
    final public WorkbenchSelectionElement wse;
    final public Object content;
    final public Resource resource;
    final public Variable variable;
    final public Object input;
    
    private WorkbenchSelectionElement extractWse(Object content) {
        if(content instanceof NodeContext) {
            NodeContext context = (NodeContext)content;
            Object input = context.getConstant(NodeType.TYPE);
            if(input instanceof NodeType) 
                return ((NodeType)input).getWorkbenchSelectionElement(context);
        }
        return null;
    }
    
    private Resource extractResource(Object content) {
    	if(content instanceof NodeContext) {
    		NodeContext context = (NodeContext)content;
    		Object input = context.getConstant(BuiltinKeys.INPUT);
    		if(input instanceof Resource) return (Resource)input;
    		if(input instanceof IAdaptable) {
    			Resource var = (Resource)((IAdaptable)input).getAdapter(Resource.class);
    			if(var != null) return var;
    		}
    	}
    	return null;
    }
    
    private Variable extractVariable(Object content) {
    	if(content instanceof NodeContext) {
    		NodeContext context = (NodeContext)content;
    		Object input = context.getConstant(BuiltinKeys.INPUT);
    		if(input instanceof Variable) return (Variable)input;
    		if(input instanceof IAdaptable) {
    			Variable var = (Variable)((IAdaptable)input).getAdapter(Variable.class);
    			if(var != null) return var;
    		}
    	}
    	return null;
    }

    private Object extractInput(Object content) {
    	if(content instanceof NodeContext) {
    		NodeContext context = (NodeContext)content;
    		return context.getConstant(BuiltinKeys.INPUT);
    	}
    	return null;
    }

    public StandardWorkbenchSelectionElement(Object content) {
        super(new Key[0]);
        this.content = content;
        this.wse = extractWse(content);
        this.resource = extractResource(content);
        this.variable = extractVariable(content);
        this.input = extractInput(content);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T getContent(WorkbenchSelectionContentType<T> contentType) {
        if (wse != null) {
            T result = wse.getContent(contentType);
            if (result != null)
                return result;
        }

        if (contentType instanceof AnyResource) {
            if (resource != null)
                return (T) resource;
            if (variable == null)
                return null;
            try {
                return (T) ((AnyResource) contentType).processor.syncRequest(new PossibleVariableRepresents(variable));
            } catch (DatabaseException e) {
                LOGGER.error("Unexpected error occurred while resolving Resource from Variable " + variable, e);
            }
        }
        else if (contentType instanceof AnyVariable) {
            if (variable != null)
                return (T) variable;
            if (resource == null)
                return null;
            try {
                return (T) ((AnyVariable) contentType).processor.syncRequest(new PossibleVariable(resource));
            } catch (DatabaseException e) {
                LOGGER.error("Unexpected error occurred while resolving Variable from Resource " + resource, e);
            }
        } else if (contentType instanceof ExplorerInputContentType) {
            return (T) input;
        }
        return null;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object getAdapter(Class adapter) {
    	if(WorkbenchSelectionElement.class == adapter) {
    		return wse;
    	}
    	if(NodeContext.class == adapter) {
    	    if(content instanceof NodeContext)
    	        return (NodeContext)content;
    	    else
    	        return null;
    	}
    	return super.getAdapter(adapter);
    }

}