/*
Copyright (c) 2012 Marco Amadei.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package net.ucanaccess.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import net.ucanaccess.commands.DDLCommandEnlist;
import net.ucanaccess.converters.Metadata;
import net.ucanaccess.converters.SQLConverter;
import net.ucanaccess.converters.SQLConverter.DDLType;
import net.ucanaccess.jdbc.FeatureNotSupportedException.NotSupportedMessage;
import net.ucanaccess.jdbc.UcanaccessSQLException.ExceptionMessages;



public abstract class AbstractExecute {
	protected enum CommandType {
		BATCH, NO_ARGUMENTS, PREPARED_STATEMENT, UPDATABLE_RESULTSET, WITH_AUTO_GENERATED_KEYS, WITH_COLUMNS_NAME, WITH_INDEXES
	}
	protected int autoGeneratedKeys;
	protected String[] columnNames;
	protected CommandType commandType;
	protected int[] indexes;
	private UcanaccessResultSet resultSet;
	protected String sql;

	private UcanaccessStatement statement;

	protected AbstractExecute(UcanaccessPreparedStatement statement) {
		this.statement = statement;
		this.commandType = CommandType.PREPARED_STATEMENT;
	}

	protected AbstractExecute(UcanaccessResultSet resultSet) {
		this.resultSet = resultSet;
		this.statement = resultSet.getWrappedStatement();
		this.commandType = CommandType.UPDATABLE_RESULTSET;
	}

	public AbstractExecute(UcanaccessStatement statement) {
		this.statement = statement;
	}

	protected AbstractExecute(UcanaccessStatement statement, String sql) {
		this(statement);
		this.commandType = CommandType.NO_ARGUMENTS;
		this.sql = sql;
	}

	protected AbstractExecute(UcanaccessStatement statement, String sql,
			int autoGeneratedKeys) {
		this(statement, sql);
		this.autoGeneratedKeys = autoGeneratedKeys;
		this.commandType = CommandType.WITH_AUTO_GENERATED_KEYS;
	}

	protected AbstractExecute(UcanaccessStatement statement, String sql,
			int[] indexes) {
		this(statement, sql);
		this.indexes = indexes;
		this.commandType = CommandType.WITH_INDEXES;
	}

	protected AbstractExecute(UcanaccessStatement statement, String sql,
			String[] columnNames) {
		this(statement, sql);
		this.columnNames = columnNames;
		this.commandType = CommandType.WITH_COLUMNS_NAME;
	}

	
	
	
	
	private Object addDDLCommand() throws SQLException {
		Object ret;
		try {
			DDLType ddlType = SQLConverter.getDDLType(sql);
			if (ddlType == null)
				throw new FeatureNotSupportedException(
						NotSupportedMessage.NOT_SUPPORTED_YET);
			
			
			String sql0=SQLConverter.convertSQL(sql).getSql();
			boolean inDis=ddlType.in(DDLType.ENABLE_AUTOINCREMENT,
					DDLType.DISABLE_AUTOINCREMENT);
			this.statement.setEnableDisable(inDis);
			if(inDis) {
				String tn=ddlType.getDBObjectName(sql0);
				if(tn==null&&sql0.indexOf('"')>0){
					tn=sql0.substring(sql0.indexOf('"')+1,sql0.lastIndexOf('"'));
				}
				UcanaccessConnection conn=(UcanaccessConnection)this.statement.getConnection();
				Metadata mtd=new Metadata(conn.getHSQLDBConnection());
				String rtn=mtd.getTableName(tn);
				if(rtn==null){
					throw new UcanaccessSQLException(ExceptionMessages.TABLE_DOESNT_EXIST, tn);
				}
				boolean inable=ddlType.equals(DDLType.ENABLE_AUTOINCREMENT);
				conn.getDbIO().getTable(rtn).setAllowAutoNumberInsert(!inable);
				return 	(this instanceof Execute) ?false:0;
			}
			String ddlExpr = ddlType.in(DDLType.CREATE_TABLE,
					DDLType.CREATE_TABLE_AS_SELECT) ? SQLConverter
					.convertCreateTable(sql0) : sql0;
				ret=	(this instanceof Execute) ?statement.getWrapped().execute(ddlExpr):statement.getWrapped().executeUpdate(ddlExpr);
		
					DDLCommandEnlist ddle = new DDLCommandEnlist();
			ddle.enlistDDLCommand(SQLConverter.restoreWorkAroundFunctions(sql), ddlType);
		} catch (Exception e) {
			throw new SQLException(e.getMessage());
		}
		return ret;
	}

	private boolean checkDDL() {
		return SQLConverter.checkDDL(this.sql);
	}

	public Object executeBase() throws SQLException {
		UcanaccessConnection conn = (UcanaccessConnection) statement
				.getConnection();
		UcanaccessConnection.setCtxConnection(conn);
		
		if(this.commandType.equals(CommandType.BATCH)){
			UcanaccessConnection.setCtxExecId(UcanaccessConnection. BATCH_ID);
		}
		else{
			UcanaccessConnection.setCtxExecId(Math.random() + "");
		}
		Object retv;
		
			if (checkDDL()) {
				retv = addDDLCommand();
			} else {
				try{
					retv = executeWrapped();
				}catch(SQLException e){
					if (conn.getAutoCommit()) {
						conn.rollback();
					}
					throw e;
				}
			}
			if (conn.getAutoCommit()) {
				conn.commit();
			}
		
		
		return retv;
	}

	public abstract Object executeWrapped() throws SQLException;

	ResultSet getWrappedResultSet() {
		return resultSet.getWrapped();
	}

	Statement getWrappedStatement() {
		return statement.getWrapped();
	}

	void setStatement(UcanaccessStatement statement) {
		this.statement = statement;
	}
}
