/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.set.hash;

import com.koloboke.collect.set.LongSet;
import com.koloboke.compile.KolobokeSet;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.*;


/**
 * This class consists only of static factory methods to construct {@code HashLongSet}s, and
 * the default {@link HashLongSetFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashLongSet
 * @see KolobokeSet @KolobokeSet
 */
public final class HashLongSets {

    private static class DefaultFactoryHolder {
        private static final HashLongSetFactory defaultFactory =
                ServiceLoader.load(HashLongSetFactory.class).iterator().next();
    }

    /**
     * Returns the default implementation of {@link HashLongSetFactory}, to which all static methods
     * in this class delegate.
     *
     * @return the default implementation of {@link HashLongSetFactory}
    
     * @throws RuntimeException if no implementations of {@link HashLongSetFactory} are provided
     */
    @Nonnull
    public static  HashLongSetFactory getDefaultFactory() {
        return (HashLongSetFactory) DefaultFactoryHolder.defaultFactory;
    }
    
    

    


    /**
     * Constructs a new empty mutable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet() newMutableSet()}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
    public static  HashLongSet newMutableSet() {
        return getDefaultFactory().newMutableSet();
    }

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(int) newMutableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
    public static  HashLongSet newMutableSet(int expectedSize) {
        return getDefaultFactory().newMutableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterable, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterator, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterator<Long> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Consumer, int) newMutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongSet newMutableSet(@Nonnull
            Consumer<com.koloboke.function.LongConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * long[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashLongSet newMutableSet(@Nonnull long[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Long[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashLongSet newMutableSet(@Nonnull Long[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterable) newMutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterable, Iterable) newMutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2) {
        return getDefaultFactory().newMutableSet(elems1, elems2);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Iterator) newMutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashLongSet newMutableSet(
            @Nonnull Iterator<Long> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Consumer) newMutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongSet newMutableSet(@Nonnull
            Consumer<com.koloboke.function.LongConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newMutableSet(elementsSupplier);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * long[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashLongSet newMutableSet(@Nonnull long[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSet(
     * Long[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashLongSet newMutableSet(@Nonnull Long[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSetOf(
     * long) newMutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
    public static  HashLongSet newMutableSetOf(long e1) {
        return getDefaultFactory().newMutableSetOf(e1);
    }

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSetOf(
     * long, long) newMutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
    public static  HashLongSet newMutableSetOf(long e1, long e2) {
        return getDefaultFactory().newMutableSetOf(e1, e2);
    }

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSetOf(
     * long, long, long) newMutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
    public static  HashLongSet newMutableSetOf(long e1, long e2, long e3) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSetOf(
     * long, long, long, long) newMutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
    public static  HashLongSet newMutableSetOf(long e1, long e2, long e3, long e4) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newMutableSetOf(long, long, long,
     * long, long, long...) newMutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
    public static  HashLongSet newMutableSetOf(long e1, long e2, long e3, long e4,
            long e5, long... restElements) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    /**
     * Constructs a new empty updatable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet() newUpdatableSet()}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet() {
        return getDefaultFactory().newUpdatableSet();
    }

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(int) newUpdatableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(int expectedSize) {
        return getDefaultFactory().newUpdatableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterable, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterator, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterator<Long> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Consumer, int) newUpdatableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(@Nonnull
            Consumer<com.koloboke.function.LongConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * long[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(@Nonnull long[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Long[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashLongSet newUpdatableSet(@Nonnull Long[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterable) newUpdatableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterable, Iterable) newUpdatableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Iterator) newUpdatableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(
            @Nonnull Iterator<Long> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Consumer) newUpdatableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(@Nonnull
            Consumer<com.koloboke.function.LongConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * long[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashLongSet newUpdatableSet(@Nonnull long[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSet(
     * Long[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashLongSet newUpdatableSet(@Nonnull Long[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSetOf(
     * long) newUpdatableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
    public static  HashLongSet newUpdatableSetOf(long e1) {
        return getDefaultFactory().newUpdatableSetOf(e1);
    }

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSetOf(
     * long, long) newUpdatableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
    public static  HashLongSet newUpdatableSetOf(long e1, long e2) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2);
    }

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSetOf(
     * long, long, long) newUpdatableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
    public static  HashLongSet newUpdatableSetOf(long e1, long e2, long e3) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSetOf(
     * long, long, long, long) newUpdatableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
    public static  HashLongSet newUpdatableSetOf(long e1, long e2, long e3, long e4) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newUpdatableSetOf(long, long, long,
     * long, long, long...) newUpdatableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
    public static  HashLongSet newUpdatableSetOf(long e1, long e2, long e3, long e4,
            long e5, long... restElements) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4, e5, restElements);
    }


    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterable, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterator, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterator<Long> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Consumer, int) newImmutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(@Nonnull
            Consumer<com.koloboke.function.LongConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * long[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(@Nonnull long[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Long[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashLongSet newImmutableSet(@Nonnull Long[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterable) newImmutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterable, Iterable) newImmutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2) {
        return getDefaultFactory().newImmutableSet(elems1, elems2);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Iterator) newImmutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(
            @Nonnull Iterator<Long> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Consumer) newImmutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(@Nonnull
            Consumer<com.koloboke.function.LongConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newImmutableSet(elementsSupplier);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * long[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashLongSet newImmutableSet(@Nonnull long[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSet(
     * Long[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashLongSet newImmutableSet(@Nonnull Long[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSetOf(
     * long) newImmutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
    public static  HashLongSet newImmutableSetOf(long e1) {
        return getDefaultFactory().newImmutableSetOf(e1);
    }

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSetOf(
     * long, long) newImmutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
    public static  HashLongSet newImmutableSetOf(long e1, long e2) {
        return getDefaultFactory().newImmutableSetOf(e1, e2);
    }

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSetOf(
     * long, long, long) newImmutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
    public static  HashLongSet newImmutableSetOf(long e1, long e2, long e3) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSetOf(
     * long, long, long, long) newImmutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
    public static  HashLongSet newImmutableSetOf(long e1, long e2, long e3, long e4) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongSetFactory#newImmutableSetOf(long, long, long,
     * long, long, long...) newImmutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
    public static  HashLongSet newImmutableSetOf(long e1, long e2, long e3, long e4,
            long e5, long... restElements) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    private HashLongSets() {}
}
