/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * 27.12.2006
 */
package org.simantics.utils.ui.gfx;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;

/**
 * HSVAdjustmentImageDescriptor
 * @author Toni Kalajainen
 */
@SuppressWarnings("deprecation")
public class HSVAdjustmentImageDescriptor extends ImageDescriptor {
    
    public static final PaletteData DEFAULT_PALETTEDATA = 
        new PaletteData(0x00ff0000, 0x0000ff00, 0x000000ff);
    
    ImageDescriptor desc;
    float h,s,v;

    /**
     * Adjust hue value
     * @param image image
     * @param hue 0..360
     * @return
     */
    public static ImageDescriptor adjustHue(ImageDescriptor image, float hue)
    {
        assert(image!=null);
        if (hue==0) return image;
        hue = mod360(hue);
        return new HSVAdjustmentImageDescriptor(image, hue, 1.0f, 1.0f);
    }
    
    /**
     * Adjust saturation
     * @param image image
     * @param saturation 0..* (1.0=normal)
     * @return
     */
    public static ImageDescriptor adjust(ImageDescriptor image, float hue, float saturation, float value)
    {
        assert(image!=null);
        if (saturation==1.0f && value==1.0f && hue==0.0f) return image;
        return new HSVAdjustmentImageDescriptor(image, hue, saturation, value);
    }

    /**
     * Adjust value
     * @param image image
     * @param value 0..* (1.0=normal)
     * @return
     */
    public static ImageDescriptor adjustSaturation(ImageDescriptor image, float saturation)
    {
        assert(image!=null);
        if (saturation==1.0f) return image;
        return new HSVAdjustmentImageDescriptor(image, 0.0f, saturation, 1.0f);
    }
    
    /**
     * Adjust hue, saturation and value
     * @param image image
     * @param hue 0..360
     * @param saturation 0..* (1.0=normal)
     * @param value 0..* (1.0=normal)
     * @return
     */
    public static ImageDescriptor adjustValue(ImageDescriptor image, float value)
    {
        assert(image!=null);
        if (value==1.0f) return image;
        return new HSVAdjustmentImageDescriptor(image, 0.0f, 1.0f, value);
    }
    
    

    /**
     * Adjusts hue, saturation and value of image
     * 
     * @param image
     * @param hue hue adjustment 0..360
     * @param saturation factor 0..* (1.0=no change)
     * @param value factor 0..* (1.0=no change)
     */
    public HSVAdjustmentImageDescriptor(ImageDescriptor image, float hue, float saturation, float value)
    {
        desc = image;
        this.h = hue;
        this.s = saturation;
        this.v = value;
    }

    @Override
    public ImageData getImageData() {
        PaletteData palette = DEFAULT_PALETTEDATA;        
        ImageData orig = ImageCache.getInstance().getImage(desc).getImageData();
        ImageData id = new ImageData(orig.width, orig.height, 24, palette);
        id.setAlpha(0,0,0);
        PaletteData origPalette = orig.palette;        
        
        ImageData mask = null;
        if (orig.getTransparencyType()==SWT.TRANSPARENCY_MASK ||
            orig.getTransparencyType()==SWT.TRANSPARENCY_PIXEL)        
            mask = orig.getTransparencyMask();
        
        for (int x=0; x<orig.width; x++)
            for (int y=0; y<orig.height; y++) {
                RGB rgb = origPalette.getRGB( orig.getPixel(x, y) );
                float hsv[] = rgb.getHSB();
                hsv[0] = mod360(hsv[0] + h);
                hsv[1] *= s;
                hsv[2] *= v;
                rgb = new RGB(hsv[0], hsv[1], hsv[2]);
                id.setPixel(x, y, palette.getPixel(rgb));
                int alpha;
                if (mask==null) 
                    alpha = orig.getAlpha(x, y);
                else
                    alpha = (mask.getPixel(x, y)==1?255:0);
                id.setAlpha(x, y, alpha);
            }        
        return id;
    }

    public static float mod360(float value)
    {
        int d;
        if (value>=0)
            d = ((int)value)/360;
        else
            d = (((int)value)/360)-1;
        return value-d*360.0f;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (obj==this) return true;
        if (!(obj instanceof HSVAdjustmentImageDescriptor))
            return false;
        HSVAdjustmentImageDescriptor other = (HSVAdjustmentImageDescriptor) obj;
        return other.desc.equals(desc) && other.h==h && other.s==s && other.v==v;        
    }
    
    @Override
    public int hashCode() {
        return desc.hashCode() ^ (new Float(h).hashCode()) ^ (new Float(s).hashCode()) ^ (new Float(v).hashCode());
    }
    
}
